// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

//       R C P C  / C R C   C L A S S E S

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
//
// P. Greg Sherwood            sherwood@code.ucsd.edu
// Kenneth Zeger	       zeger@code.ucsd.edu
//
// 9500 Gilman Dr MC 0407
// La Jolla, CA 92093
//
//
// Copyright (c) 1997 P. Greg Sherwood and Kenneth Zeger
//
// This program is Copyright (c) by P. Greg Sherwood and Kenneth Zeger
// It may not be redistributed without written permission of its 
// copyright holders. It may not be sold for profit or
// incorporated in commercial programs without the written permission
// of the copyright holders. This program is provided as is, without any
// express or implied warranty, without even the warranty of fitness
// for a particular purpose.
//

#include "rcpc_crc.h"
#include "general.h"
#include "fec.h"
#include "rcpc_code_lib.h"

static char encclassname[] = "RcpcCRCEncoder";
static char decclassname[] = "RcpcCRCDecoder";
const int CRC_SIZE = 16;
const int DefaultInfoBits = 200;

static void crc_setup(int packet_size, int *reduced_table);

void crc_setup(int packet_size, int* reduced_table)
{
  // CRC generator definitions
#define CRC_C1_GEN 0xCB6A
#define CRC_C2_GEN 0xF49F
#define CRC_C3_GEN 0xAC9A
#define CRC_C4_GEN 0x088D
#define CRC_C5_GEN 0x547D
#define CRC_ANSI_GEN  0xA001// Used in CRC-16 standard
#define CRC_CCITT_GEN 0x8048
#define CRC_IBM_SDLC_GEN 0x509E


// -------------------------------------------------------------------

  // The following CRC polynomials are from Castagnoli, Ganz, and Graber, "Optimum
  // Cyclic Redundancy-Check Codes with 16-Bit Redundancy," pp. 111-- 114, 
  // IEEE_COM, vol 38, no. 1, Jan. 1990.

  if (packet_size <= 151)
    { // Use code C1 of Table II p. 113
      compute_reduced_table(CRC_C1_GEN, reduced_table);
    }
  else if (packet_size > 151 && packet_size <= 257)
    {
      compute_reduced_table(CRC_C3_GEN, reduced_table);
    }
  else if (packet_size > 257 && packet_size <= 28658)
    {
      compute_reduced_table(CRC_C4_GEN, reduced_table);
    }
  else
    {
      compute_reduced_table(CRC_C5_GEN, reduced_table);
    }

}

// -------------------------------------------------------------------



// *******************************************************************
// *
// *                   RcpcCRC Encoder Class
// *
// *
// *******************************************************************


RcpcCRCEncoder::RcpcCRCEncoder(int in_info_bits, int in_code_spec_num, 
			       int in_code_num)
{
  init_values(in_info_bits, in_code_spec_num, 
	      in_code_num);
}

void RcpcCRCEncoder::init_values(int in_info_bits, int in_code_spec_num, 
				 int in_code_num)
{
  int packet_size;
  
  rcpc_code_spec_num = in_code_spec_num;
  rcpc_code_num = in_code_num;

  info_bits = in_info_bits;
  if (in_info_bits <= 0)
    {
      ErrorMsg(encclassname, "init_values", "Invalid info bits value of %d - setting to default value of %d\n", in_info_bits, DefaultInfoBits);
      info_bits = DefaultInfoBits;
    }

  // create internal objects 
  if (rcpc_code_spec_num > CRC_ONLY)
    {
      RCPCCodeSpec code_spec = get_rcpc_codespec(rcpc_code_spec_num);
      rcpc_encoder = new RCPCEncoder(&code_spec);
      rcpc_encoder->select_code(rcpc_code_num);
      // initialize CRC 
      packet_size =  rcpc_encoder->GetMemSize() + CRC_SIZE + info_bits;
      crc_setup(packet_size, reduced_table);

    }
  else if (rcpc_code_spec_num == CRC_ONLY)
    {
      rcpc_encoder = NULL;
      // initialize CRC 
      packet_size =  CRC_SIZE + info_bits;
      crc_setup(packet_size, reduced_table);
    }
  else
    rcpc_encoder = NULL;


}

// -------------------------------------------------------------------

RcpcCRCEncoder::RcpcCRCEncoder(const RcpcCRCEncoder& src)
{
  copy_values(src);
}

// -------------------------------------------------------------------

RcpcCRCEncoder& RcpcCRCEncoder::operator =(const RcpcCRCEncoder& src)
{
  free_memory();
  copy_values(src);
  return *this;
}

// -------------------------------------------------------------------

void RcpcCRCEncoder::copy_values(const RcpcCRCEncoder& src)
{
  info_bits = src.info_bits;
  memcpy(reduced_table, src.reduced_table, sizeof(reduced_table));
  rcpc_code_spec_num = src.rcpc_code_spec_num;
  rcpc_code_num = src.rcpc_code_num;

  if (rcpc_code_spec_num > CRC_ONLY)
    {
      RCPCCodeSpec code_spec = get_rcpc_codespec(rcpc_code_spec_num);
      rcpc_encoder = new RCPCEncoder(&code_spec);
      rcpc_encoder->select_code(rcpc_code_num);
    }
  else 
    rcpc_encoder = NULL;


}


// -------------------------------------------------------------------

// Destructor
RcpcCRCEncoder::~RcpcCRCEncoder()
{
  free_memory();
}

// -------------------------------------------------------------------

void RcpcCRCEncoder::free_memory()
{
  if (rcpc_encoder)
    delete rcpc_encoder;
}

// -------------------------------------------------------------------

int RcpcCRCEncoder::get_chan_bits()
{
  if (rcpc_code_spec_num > CRC_ONLY)
    return rcpc_encoder->compute_coded_bits(rcpc_encoder->GetMemSize() 
					    + CRC_SIZE + info_bits);
  else if (rcpc_code_spec_num == CRC_ONLY)
    return info_bits + CRC_SIZE;
  else
    return info_bits;
}

// -------------------------------------------------------------------

RcpcCRCEncodeStatus RcpcCRCEncoder::encode(BitBuffer& in_buffer, 
					   BitBuffer& out_buffer)
{
  int crc;
  BitBuffer tmpbuf(1);

  if (in_buffer.rbits_to_end() < info_bits)
    return RcpcCRCEncodeInsufficientInfoBits;

  move_bits(&tmpbuf, &in_buffer, 
	    info_bits);

  tmpbuf.position_read_ptr(0);

  if (rcpc_code_spec_num >= CRC_ONLY)
    {
      crc = compute_crc_16(&tmpbuf, info_bits, reduced_table);
      tmpbuf.buffer_bits(CRC_SIZE, crc);
    }
  
  if (rcpc_code_spec_num > CRC_ONLY)
    {  // encode with rcpc code
      rcpc_encoder->reset();
      tmpbuf.position_read_ptr(0);
      rcpc_encoder->encode(tmpbuf.bits_in_buffer(), 
			   &tmpbuf, &out_buffer);
      rcpc_encoder->flush_rcpc_mem(&out_buffer);
    }
  else
    {
      tmpbuf.position_read_ptr(0);
      move_bits(&out_buffer, &tmpbuf);
    }

  return RcpcCRCEncodeSuccess;

}

// -------------------------------------------------------------------

// ===================================================================
// =
// =         Header Support Routines
// =
// =
// ===================================================================

void RcpcCRCEncoder::get_header(RcpcCRCHeader& hdr)
{
  hdr.info_bits = info_bits;
  hdr.rcpc_code_spec_num = rcpc_code_spec_num;
  hdr.rcpc_code_num = rcpc_code_num;

}

// -------------------------------------------------------------------

void RcpcCRCEncoder::dump_header(BitBuffer& outbuf)
{
  RcpcCRCHeader hdr;

  get_header(hdr);
  hdr.dump(outbuf);
}

// -------------------------------------------------------------------

ostream& operator << (ostream& s, const RcpcCRCHeader& hdr)
{
  s << "The RCPC/CRC (RcpcCRC) code header contains:" << endl;
  s << "Info bits = " << hdr.info_bits 
    << " RCPC code family = " << hdr.rcpc_code_spec_num 
    << " RCPC code number = " << hdr.rcpc_code_num << endl;
  return s;
}

// -------------------------------------------------------------------
void RcpcCRCHeader::dump(BitBuffer& out_buf)
{
  // Dump the values to the bitbuffer with 
  // proper bit resolution
  out_buf.buffer_bits(RcpcCRCHdrSizes[1], info_bits);
  out_buf.buffer_bits(RcpcCRCHdrSizes[2], rcpc_code_spec_num);
  out_buf.buffer_bits(RcpcCRCHdrSizes[3], rcpc_code_num);
  
}


// -------------------------------------------------------------------

int RcpcCRCHeader::fill(BitBuffer& in_buf)
{
  if (in_buf.rbits_to_end() < RcpcCRCHdrSizes[0])
    {
      return 0;
    }
  
  in_buf.read_bits(RcpcCRCHdrSizes[1], info_bits);
  in_buf.read_bits(RcpcCRCHdrSizes[2], rcpc_code_spec_num);
  in_buf.read_bits(RcpcCRCHdrSizes[3], rcpc_code_num);
  return 1;

}

// -------------------------------------------------------------------
// *******************************************************************
// *
// *                   RcpcCRC Decoder Class
// *
// *
// *******************************************************************

RcpcCRCDecoder::RcpcCRCDecoder(int in_info_bits, int in_code_spec_num, 
			       int in_code_num)
{
  // fill in header structure
  RcpcCRCHeader hdr;
  
  hdr.info_bits = in_info_bits;
  hdr.rcpc_code_spec_num = in_code_spec_num;
  hdr.rcpc_code_num = in_code_num;
  
  setup(hdr);

}

// -------------------------------------------------------------------

RcpcCRCDecoder::RcpcCRCDecoder(RcpcCRCHeader& header)
{
  setup(header);
}

// -------------------------------------------------------------------

RcpcCRCDecoder::RcpcCRCDecoder(const RcpcCRCDecoder& src)
{
  copy_values(src);
}

// -------------------------------------------------------------------

void RcpcCRCDecoder::setup(RcpcCRCHeader& hdr)
{
  int packet_size;

  info_bits = hdr.info_bits;
  rcpc_code_spec_num = hdr.rcpc_code_spec_num;
  rcpc_code_num = hdr.rcpc_code_num;

  if (info_bits <= 0)
    {
      ErrorMsg(decclassname, "setup", "Invalid info bits value of %d - setting to default value of %d\n", info_bits, DefaultInfoBits);
      info_bits = DefaultInfoBits;
    }

  if (rcpc_code_spec_num > CRC_ONLY)
    {  // create internal objects 
      RCPCCodeSpec code_spec = get_rcpc_codespec(rcpc_code_spec_num);
      rcpc_decoder = new RCPCDecoder(&code_spec, NULL, rcpc_code_num, 0, 1);
      rcpc_decoder->select_code(rcpc_code_num);
      // initialize CRC 
      packet_size =  rcpc_decoder->GetMemSize() + CRC_SIZE + info_bits;
      crc_setup(packet_size, reduced_table);
      chan_bits = rcpc_decoder->compute_coded_bits(packet_size);
    }
  else if (rcpc_code_spec_num == CRC_ONLY)
    {
      rcpc_decoder = NULL;
      chan_bits = packet_size =  CRC_SIZE + info_bits;
      crc_setup(packet_size, reduced_table);
    }
  else
    {
      rcpc_decoder = NULL;
      chan_bits = info_bits;
    }

  VerboseMsg(decclassname, "setup", 3, 
	     "chan bits = %d\n", chan_bits);

}

// -------------------------------------------------------------------

void RcpcCRCDecoder::copy_values(const RcpcCRCDecoder& src)
{
  info_bits = src.info_bits;
  chan_bits = src.chan_bits;
  memcpy(reduced_table, src.reduced_table, sizeof(reduced_table));
  rcpc_code_spec_num = src.rcpc_code_spec_num;
  rcpc_code_num = src.rcpc_code_num;

  if (rcpc_code_spec_num > CRC_ONLY)
    {
      RCPCCodeSpec code_spec = get_rcpc_codespec(rcpc_code_spec_num);
      rcpc_decoder = new RCPCDecoder(&code_spec, NULL, rcpc_code_num, 0, 1);
      rcpc_decoder->select_code(rcpc_code_num);
    }
  else
    rcpc_decoder = NULL;

}


// -------------------------------------------------------------------

RcpcCRCDecoder& RcpcCRCDecoder::operator =(const RcpcCRCDecoder& src)
{
  free_memory();
  copy_values(src);
  return *this;
}

// -------------------------------------------------------------------

// Destructor
RcpcCRCDecoder::~RcpcCRCDecoder()
{
  free_memory();
}

// -------------------------------------------------------------------

void RcpcCRCDecoder::free_memory()
{
  if (rcpc_decoder)
    delete rcpc_decoder;
}

// -------------------------------------------------------------------

RcpcCRCDecodeStatus RcpcCRCDecoder::decode(BitBuffer& in_buffer, 
					   BitBuffer& out_buffer, 
					   int srch_depth)
{
  int crc, dec_crc;
  int erasure_flag;
  BitBuffer tmp_chanbuf(1);
  BitBuffer tmp_decode_buf(1);

  srch_depth = (srch_depth > 0) ? srch_depth : DefaultTrellisSearchDepth;

  if (in_buffer.rbits_to_end() < chan_bits)
    return RcpcCRCDecodeInsufficientChanBits;

  move_bits(&tmp_chanbuf, &in_buffer, chan_bits);

  erasure_flag = 0;

  tmp_chanbuf.position_read_ptr(0);
  if (rcpc_code_spec_num > CRC_ONLY)
    {
      rcpc_decoder->reset_trellis();
      rcpc_decoder->clear_tt_stack();
      
      rcpc_decoder->decode_bits(chan_bits, 
				&tmp_chanbuf, &tmp_decode_buf);
      rcpc_decoder->terminate_trellis(&tmp_decode_buf,srch_depth);
      
      tmp_decode_buf.position_read_ptr(0);
      dec_crc = compute_crc_16(&tmp_decode_buf, info_bits, reduced_table);
      
      tmp_decode_buf.position_read_ptr(info_bits);
      tmp_decode_buf.read_bits(CRC_SIZE, crc);
      
      if (crc != dec_crc)
	{
	  int jj;
	  
	  for (jj = 1; jj < srch_depth; ++jj)
	    {
	      // empty out buffer
	      tmp_decode_buf.empty_buffer();
	      rcpc_decoder->decode_nth_candidate(jj, &tmp_decode_buf);
	      dec_crc = compute_crc_16(&tmp_decode_buf, 
				       info_bits, reduced_table);
	      tmp_decode_buf.position_read_ptr(info_bits);
	      tmp_decode_buf.read_bits(CRC_SIZE, crc);
	      if (crc == dec_crc)
		{
		  VerboseMsg(decclassname, "decode", 2, 
			     "Found CRC match for candidate %d\n", jj+1);
		  break;
		}
	    }
	  if (jj == srch_depth)
	    {
	      VerboseMsg(decclassname, "decode", 3, 
			 "NO CRC match found in top %d paths\n",
			 srch_depth);
	      // just decode the top candidate and compute the number of errors
	      tmp_decode_buf.empty_buffer();
	      rcpc_decoder->decode_nth_candidate(0, &tmp_decode_buf);
	      tmp_decode_buf.position_read_ptr(0);
	      
	      erasure_flag = 1;
	    }
	}
      tmp_decode_buf.position_read_ptr(0);
      move_bits(&out_buffer, &tmp_decode_buf,
		info_bits);

    }

  else if (rcpc_code_spec_num == CRC_ONLY)
    {
      tmp_chanbuf.position_read_ptr(0);
      dec_crc = compute_crc_16(&tmp_chanbuf, info_bits, reduced_table);
      
      tmp_chanbuf.position_read_ptr(info_bits);
      tmp_chanbuf.read_bits(CRC_SIZE, crc);
      if (dec_crc != crc) erasure_flag = 1;

      tmp_chanbuf.position_read_ptr(0);
      move_bits(&out_buffer, &tmp_chanbuf,
		info_bits);
	
    }
  else
    {
      tmp_chanbuf.position_read_ptr(0);
      move_bits(&out_buffer, &tmp_chanbuf,
		info_bits);
    }

  if (erasure_flag)
    return RcpcCRCDecodeNoCRCMatch;

  return RcpcCRCDecodeSuccess;

}



// -------------------------------------------------------------------

void RcpcCRCDecoder::get_header(RcpcCRCHeader& hdr)
{
  hdr.info_bits = info_bits;
  hdr.rcpc_code_spec_num = rcpc_code_spec_num;
  hdr.rcpc_code_num = rcpc_code_num;

}


