// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

//       R C P C  C O D E  L I B

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
//
// P. Greg Sherwood            sherwood@code.ucsd.edu
// Kenneth Zeger	       zeger@code.ucsd.edu
//
// 9500 Gilman Dr MC 0407
// La Jolla, CA 92093
//
//
// Copyright (c) 1997 P. Greg Sherwood and Kenneth Zeger
//
// This program is Copyright (c) by P. Greg Sherwood and Kenneth Zeger
// It may not be redistributed without written permission of its 
// copyright holders. It may not be sold for profit or
// incorporated in commercial programs without the written permission
// of the copyright holders. This program is provided as is, without any
// express or implied warranty, without even the warranty of fitness
// for a particular purpose.
//

#include "rcpc_code_lib.h"

// This file contains code definitions for several
// RCPC convolutional codes.

// NOTE: The code structures below contain information
// in the following order:
//     BaseNout, MemSize, Period, NumCodes, *GenMat, *PunctMat
// As specified for the structure RCPCCodeSpec in rcpc.h

/**************************************************
 * The following codes and puncturing matrices 
 * are from:
 * J. Hagenauer "Rate Compatible Convolutional Codes 
 *     and their Applications," IEEE Trans. Comm., vol. 36
 *     No. 4, April 1988, pp. 389-400.
 **************************************************/

/*************************************************
 * Code 1 is based on a rate 1/4 memory 4 mother
 * code with 13 different rates available in the 
 * puncturing matrix of period 8. 
 *************************************************/
#define HagenGenMat1 {0x19, 0x17, 0x1D, 0x1B}
#define HagenCode1 4, 4, 8, 13
#define HagenPunctMat1 { \
  0xC, 0x8, 0x8, 0x8, 0x4, 0x8, 0x8, 0x8,\
  0xC, 0x8, 0x8, 0x8, 0xC, 0x8, 0x8, 0x8,\
  0xC, 0x8, 0xC, 0x8, 0xC, 0x8, 0xC, 0x8,\
  0xC, 0xC, 0xC, 0x8, 0xC, 0xC, 0xC, 0x8,\
  0xC, 0xC, 0xC, 0xC, 0xC, 0xC, 0xC, 0xC,\
  0xE, 0xC, 0xC, 0xC, 0xE, 0xC, 0xC, 0xC,\
  0xE, 0xE, 0xC, 0xC, 0xE, 0xE, 0xC, 0xC,\
  0xE, 0xE, 0xE, 0xC, 0xE, 0xE, 0xE, 0xC,\
  0xE, 0xE, 0xE, 0xE, 0xE, 0xE, 0xE, 0xE,\
  0xF, 0xE, 0xE, 0xE, 0xF, 0xE, 0xE, 0xE,\
  0xF, 0xE, 0xF, 0xE, 0xF, 0xE, 0xF, 0xE,\
  0xF, 0xF, 0xF, 0xE, 0xF, 0xF, 0xF, 0xE,\
  0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF\
}



/*************************************************
 * Code 2 is based on a rate 1/3 memory 5 mother
 * code with 9 different rates available in the 
 * puncturing matrix of period 8. 
 *************************************************/
#define HagenGenMat2 {0x2F, 0x35, 0x39}
#define HagenCode2  3, 5, 8, 9
#define HagenPunctMat2  {\
  0x6, 0x4, 0x4, 0x4, 0x4, 0x4, 0x4, 0x4,\
  0x6, 0x4, 0x4, 0x4, 0x6, 0x4, 0x4, 0x4,\
  0x6, 0x4, 0x6, 0x4, 0x6, 0x4, 0x6, 0x4,\
  0x6, 0x6, 0x6, 0x4, 0x6, 0x6, 0x6, 0x4,\
  0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6,\
  0x7, 0x6, 0x6, 0x6, 0x7, 0x6, 0x6, 0x6,\
  0x7, 0x6, 0x7, 0x6, 0x7, 0x6, 0x7, 0x6,\
  0x7, 0x7, 0x7, 0x6, 0x7, 0x7, 0x7, 0x6,\
  0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7\
}


/*************************************************
 * Code 3 is based on a rate 1/3 memory 6 mother
 * code with 9 different rates available in the 
 * puncturing matrix of period 8. 
 *************************************************/
#define HagenGenMat3 {0x6D, 0x4F, 0x53}
#define HagenCode3  3, 6, 8, 9
#define HagenPunctMat3 { \
  0x6, 0x4, 0x4, 0x4, 0x4, 0x4, 0x4, 0x4,\
  0x6, 0x4, 0x4, 0x4, 0x6, 0x4, 0x4, 0x4,\
  0x6, 0x4, 0x6, 0x4, 0x6, 0x4, 0x6, 0x4,\
  0x6, 0x6, 0x6, 0x4, 0x6, 0x6, 0x6, 0x4,\
  0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6,\
  0x7, 0x6, 0x6, 0x6, 0x7, 0x6, 0x6, 0x6,\
  0x7, 0x7, 0x6, 0x6, 0x7, 0x7, 0x6, 0x6,\
  0x7, 0x7, 0x7, 0x6, 0x7, 0x7, 0x7, 0x6,\
  0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7\
}


/**************************************************
 * The following codes and puncturing matrices 
 * are from:
 * L. H. C. Lee "New Rate-Compatible Punctured Convolutional Codes 
 *     for Viterbi Decoding," IEEE Trans. Comm., vol. 42
 *     No. 12, December 1994, pp. 3073-3079.
 **************************************************/

/*************************************************
 * Code 1 is based on a rate 1/2 memory 4 mother
 * code with 6 different rates available in the 
 * puncturing matrix of period 7. 
 *************************************************/
#define LeeGenMat1 {0x13, 0x1D}
#define LeeCode1  2, 4, 7, 6
#define LeePunctMat1 { \
  0x2, 0x3, 0x2, 0x2, 0x2, 0x1, 0x1,\
  0x2, 0x3, 0x2, 0x2, 0x2, 0x3, 0x1,\
  0x2, 0x3, 0x2, 0x3, 0x2, 0x3, 0x1,\
  0x2, 0x3, 0x2, 0x3, 0x2, 0x3, 0x3,\
  0x3, 0x3, 0x2, 0x3, 0x2, 0x3, 0x3,\
  0x3, 0x3, 0x2, 0x3, 0x3, 0x3, 0x3\
}

/*************************************************
 * Code 2 is based on a rate 1/2 memory 5 mother
 * code with 6 different rates available in the 
 * puncturing matrix of period 7. 
 *************************************************/
#define LeeGenMat2 {0x2B, 0x3D}
#define LeeCode2  2, 5, 7, 6
#define LeePunctMat2 { \
  0x2, 0x3, 0x2, 0x2, 0x2, 0x1, 0x2,\
  0x2, 0x3, 0x2, 0x2, 0x2, 0x1, 0x3,\
  0x2, 0x3, 0x2, 0x3, 0x2, 0x1, 0x3,\
  0x2, 0x3, 0x2, 0x3, 0x2, 0x3, 0x3,\
  0x3, 0x3, 0x2, 0x3, 0x2, 0x3, 0x3,\
  0x3, 0x3, 0x3, 0x3, 0x2, 0x3, 0x3\
}


/*************************************************
 * Code 3 is based on a rate 1/2 memory 6 mother
 * code with 6 different rates available in the 
 * puncturing matrix of period 7. 
 *************************************************/
#define LeeGenMat3 {0x5B, 0x79}
#define LeeCode3  2, 6, 7, 6
#define LeePunctMat3 { \
  0x3, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2,\
  0x3, 0x2, 0x2, 0x2, 0x3, 0x2, 0x2,\
  0x3, 0x2, 0x2, 0x2, 0x3, 0x3, 0x2,\
  0x3, 0x2, 0x3, 0x2, 0x3, 0x3, 0x2,\
  0x3, 0x3, 0x3, 0x2, 0x3, 0x3, 0x2,\
  0x3, 0x3, 0x3, 0x2, 0x3, 0x3, 0x3\
}

#define NewGenMat1 {0x35, 0x3B, 0x27, 0x2F}
#define NewCode1 4, 5, 8, 24
#define NewPunctMat1 { \
  0xC, 0x8, 0x8, 0x8, 0x8, 0x2, 0x8, 0x8, \
  0xE, 0x8, 0x8, 0x8, 0x8, 0x2, 0x8, 0x8, \
  0xF, 0x8, 0x8, 0x8, 0x8, 0x2, 0x8, 0x8, \
  0xF, 0x8, 0x8, 0x8, 0x8, 0xA, 0x8, 0x8, \
  0xF, 0x8, 0x8, 0x8, 0x8, 0xA, 0x8, 0xC, \
  0xF, 0x8, 0x8, 0x8, 0x8, 0xE, 0x8, 0xC, \
  0xF, 0xC, 0x8, 0x8, 0x8, 0xE, 0x8, 0xC, \
  0xF, 0xC, 0x8, 0xC, 0x8, 0xE, 0x8, 0xC, \
  0xF, 0xC, 0x8, 0xC, 0x8, 0xE, 0x8, 0xE, \
  0xF, 0xC, 0x8, 0xC, 0x8, 0xE, 0xC, 0xE, \
  0xF, 0xC, 0xA, 0xC, 0x8, 0xE, 0xC, 0xE, \
  0xF, 0xC, 0xA, 0xC, 0x8, 0xE, 0xE, 0xE, \
  0xF, 0xE, 0xA, 0xC, 0x8, 0xE, 0xE, 0xE, \
  0xF, 0xE, 0xA, 0xC, 0xC, 0xE, 0xE, 0xE, \
  0xF, 0xE, 0xA, 0xC, 0xC, 0xE, 0xF, 0xE, \
  0xF, 0xE, 0xA, 0xC, 0xE, 0xE, 0xF, 0xE, \
  0xF, 0xE, 0xA, 0xC, 0xE, 0xF, 0xF, 0xE, \
  0xF, 0xE, 0xE, 0xC, 0xE, 0xF, 0xF, 0xE, \
  0xF, 0xE, 0xE, 0xD, 0xE, 0xF, 0xF, 0xE, \
  0xF, 0xE, 0xE, 0xF, 0xE, 0xF, 0xF, 0xE, \
  0xF, 0xE, 0xF, 0xF, 0xE, 0xF, 0xF, 0xE, \
  0xF, 0xE, 0xF, 0xF, 0xE, 0xF, 0xF, 0xF, \
  0xF, 0xE, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, \
  0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, \
}

// {0x5D, 0x5D, 0x73, 0x67}
#define NewGenMat2 {0x6D, 0x53, 0x5F, 0x67}
#define NewCode2 4, 6, 8, 23
#define NewPunctMat2 { \
0x8, 0xA, 0x8, 0xA, 0x8, 0x8, 0x2, 0x8, \
0x8, 0xA, 0x8, 0xA, 0x8, 0x8, 0xA, 0x8, \
0x8, 0xA, 0x8, 0xA, 0x8, 0xC, 0xA, 0x8, \
0x8, 0xA, 0x8, 0xA, 0xC, 0xC, 0xA, 0x8, \
0x8, 0xA, 0x8, 0xA, 0xC, 0xC, 0xE, 0x8, \
0x8, 0xA, 0x8, 0xA, 0xC, 0xE, 0xE, 0x8, \
0x8, 0xE, 0x8, 0xA, 0xC, 0xE, 0xE, 0x8, \
0xC, 0xE, 0x8, 0xA, 0xC, 0xE, 0xE, 0x8, \
0xC, 0xE, 0x8, 0xE, 0xC, 0xE, 0xE, 0x8, \
0xE, 0xE, 0x8, 0xE, 0xC, 0xE, 0xE, 0x8, \
0xC, 0xE, 0xC, 0xE, 0xC, 0xE, 0xC, 0xE, \
0xC, 0xE, 0xC, 0xE, 0xE, 0xE, 0xC, 0xE, \
0xE, 0xE, 0xC, 0xE, 0xE, 0xE, 0xC, 0xE, \
0xE, 0xE, 0xC, 0xE, 0xE, 0xE, 0xE, 0xE, \
0xE, 0xE, 0xE, 0xE, 0xE, 0xE, 0xE, 0xE, \
0xE, 0xE, 0xE, 0xE, 0xE, 0xF, 0xE, 0xE, \
0xE, 0xF, 0xE, 0xE, 0xE, 0xF, 0xE, 0xE, \
0xE, 0xF, 0xE, 0xE, 0xE, 0xF, 0xE, 0xF, \
0xE, 0xF, 0xE, 0xF, 0xE, 0xF, 0xE, 0xF, \
0xE, 0xF, 0xE, 0xF, 0xF, 0xF, 0xE, 0xF, \
0xF, 0xF, 0xE, 0xF, 0xF, 0xF, 0xE, 0xF, \
0xF, 0xF, 0xE, 0xF, 0xF, 0xF, 0xF, 0xF, \
0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, \
}


static int hgen1[] = HagenGenMat1;
static int hgen2[] = HagenGenMat2;
static int hgen3[] = HagenGenMat3;
static int hpunc1[] = HagenPunctMat1;
static int hpunc2[] = HagenPunctMat2;
static int hpunc3[] = HagenPunctMat3;

static int lgen1[] = LeeGenMat1;
static int lgen2[] = LeeGenMat2;
static int lgen3[] = LeeGenMat3;
static int lpunc1[] = LeePunctMat1;
static int lpunc2[] = LeePunctMat2;
static int lpunc3[] = LeePunctMat3;

static int ngen1[] = NewGenMat1;
static int npunc1[] = NewPunctMat1;
static int ngen2[] = NewGenMat2;
static int npunc2[] = NewPunctMat2;

static RCPCCodeSpec CodeSpecTable[] = {
  {HagenCode1, hgen1, hpunc1},
  {HagenCode2, hgen2, hpunc2},
  {HagenCode3, hgen3, hpunc3},
  {NewCode1, ngen1, npunc1},
  {NewCode2, ngen2, npunc2},
  {LeeCode1, lgen1, lpunc1},
  {LeeCode2, lgen2, lpunc2},
  {LeeCode3, lgen3, lpunc3}};

RCPCCodeSpec get_rcpc_codespec(int codespec_num)
{
  if (codespec_num < 0 || 
      codespec_num >= ((sizeof(CodeSpecTable))/sizeof(CodeSpecTable[0])))
    codespec_num = 0;
  
  return CodeSpecTable[codespec_num];
}


