// -*- c++ -*-
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

//       R C P C   ( E N / D E ) C O D E R   C L A S S E S

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
//
// P. Greg Sherwood            sherwood@code.ucsd.edu
// Kenneth Zeger	       zeger@code.ucsd.edu
//
// 9500 Gilman Dr MC 0407
// La Jolla, CA 92093
//
//
// Copyright (c) 1997 P. Greg Sherwood and Kenneth Zeger
//
// This program is Copyright (c) by P. Greg Sherwood and Kenneth Zeger
// It may not be redistributed without written permission of its 
// copyright holders. It may not be sold for profit or
// incorporated in commercial programs without the written permission
// of the copyright holders. This program is provided as is, without any
// express or implied warranty, without even the warranty of fitness
// for a particular purpose.
//


#ifndef __RCPC_H
#define __RCPC_H

// - - Inclusion - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#include <stdio.h>
#include <string.h>
#include "general.h"
#include "bitbuffer.h"
#include "fec.h"


#define INIT_METRIC 1E+10  /* an initial very large number */
#define TRELLIS_INIT 1
#define TRELLIS_FLUSH 2
#define TRELLIS_NORM 3

// RateEntry type
typedef struct __SchedEntryType {
  int location;
  int code_num;
} SchedEntryType;

// Rate schedule definition
typedef struct __RateSchedType {
  int size;
  SchedEntryType *sched_array;
} RateSchedType;

typedef struct __RCPCCodeSpec {
  int BaseNout;
  int MemSize;
  int Period;
  int NumCodes;
  int *GenMat;
  int *PunctMat;
} RCPCCodeSpec;



// - - External function - - - - - - - - - - - - - - - - - - - - - - - -


// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

//  Class definitions

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

class RCPCEncoder
{
  // . class data   .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .

  protected :

  
  int oindex, per_idx, rcpc_buffer;
  int code_num;
  int punc_entry;

  int sched_idx;
  int loc;
  RateSchedType *rate_sched;
  RCPCCodeSpec code_spec;

  int *punc_ptr;

  int rcpc_mem_full;
  
  int send_bit();
  

  // . constructor  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .

  public :

  RCPCEncoder(RCPCCodeSpec *in_code_spec = NULL, 
	      RateSchedType *in_rate_sched=NULL, int code_num = 0);
  RCPCEncoder(RateSchedType *in_rate_sched=NULL, int code_num = 0);
  ~RCPCEncoder();

  // . class functions .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .

  int GetMemSize();

  int get_codenum() {return code_num;};

  void reset(int state = 0, int punc_phase = 0);

  void set_state(int state);

  int compute_coded_bits(int raw_bits, int code_num);

  int compute_coded_bits(int raw_bits);

  int compute_src_bits(int coded_bits, int code_num);

  int compute_src_bits(int coded_bits);

  void encode(int bits, int *input, BitBuffer *buffer);

  int encode(int bits, BitBuffer *input, BitBuffer *buffer);

  int max_code_index(void);

  int flush_rcpc_mem(BitBuffer *buffer);

  int set_rate_sched(RateSchedType *in_rate_sched);

  void select_code(int code_num);

}; // end of class < RCPCEncoder >


// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

struct SurvivorType 
{
  int state;
  float metric;
  SurvivorType *backptr;

  SurvivorType() {state = 0; metric = 0; backptr = NULL;};
  SurvivorType(int in_state, float in_metric, SurvivorType* ptr) {state = in_state; metric = in_metric; backptr = ptr;};
};

typedef struct _TTPathSegment {
  SurvivorType *start;
  int length;
  struct _TTPathSegment *next;
} TTPathSegment;


struct TTStack {
  float metric;
  TTPathSegment *path;

  TTStack() {metric = 0; path=NULL;};
  TTStack(float in_metric, TTPathSegment* in_path) {metric = in_metric; path = in_path;};
};


#define DEFAULT_SEGSIZE 100
#define DEFAULT_PATHBUF_SIZE 100

class RCPCDecoder
{
  // . class data   .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .

  protected :
  
  SurvivorType ***path_buffer;
  SurvivorType *working_trellis_stage;

  TTStack *tt_stack;
  int tt_stack_size;
  int tt_last_path_found;

  int tt_mode;
  
  int code_num, *punc_ptr;

  int punc_entry, per_idx;
  int alloc_pathbuf_idx, alloc_pathseg_idx, pathbuf_idx, pathseg_idx;

  int pathbuf_size, pathseg_size, *labels, num_states, num_stages, stage_size;
  int current, bits_for_transition;
  
  int *token_buffer, *new_token, buffer_full, state_mask;

  int sched_idx;
  int loc;
  RateSchedType *rate_sched;
  RCPCCodeSpec code_spec;

  // variables for decoding concatenated code.

  int *parity_array, *working_parity_array;
  int parity_start, inner_phase, inner_period;

  
  virtual float compute_normalization(int old_state, int recvd_word) {old_state=old_state; recvd_word=recvd_word;return 0.0;};

  virtual float compute_metric(int recvd, unsigned int punc, int label, 
			       int start_state, int end_state);
  
  void clear_trellis();


  // . constructor  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .

  public :

  RCPCDecoder(RCPCCodeSpec *in_code_spec = NULL,
	      RateSchedType *in_rate_sched = NULL,
	      int code_number = 0, int trellis_stages = 0,
	      int tree_trellis_mode = 0,
	      int buf_size = DEFAULT_PATHBUF_SIZE,
	      int seg_size = DEFAULT_SEGSIZE);
  RCPCDecoder(RateSchedType *in_rate_sched = NULL,
	      int code_number = 0, int trellis_stages = 0,
	      int tree_trellis_mode = 0,
	      int buf_size = DEFAULT_PATHBUF_SIZE,
	      int seg_size = DEFAULT_SEGSIZE);

  ~RCPCDecoder();

  // . class functions .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .


  void recurse_buffer(SurvivorType *ptr, int level, BitBuffer *out_buffer);

  void reset_memory();

  void reset_trellis();

  void terminate_trellis(BitBuffer *out_buffer, int num_candidates = 1);

  int decode_bits(int word, int bits, BitBuffer *out_buffer, 
		  int flag = TRELLIS_NORM);

  int decode_bits(int bits, BitBuffer *input, BitBuffer *out_buffer, 
		  int flag = TRELLIS_NORM);

  void decode_pending(BitBuffer *out_buffer);

  int max_code_index(void);

  int compute_coded_bits(int raw_bits, int code_num);

  int compute_coded_bits(int raw_bits);

  void select_code(int code_num);

  void decode_TTpath(TTPathSegment *TTptr, 
		     BitBuffer *out_buffer,
		     int num_bits = 0 );

  int decode_nth_candidate(int num, BitBuffer *out_buffer);

  void clear_tt_stack();

  void init_tt_stack(int num_elements);

  void find_multiple_candidates(int num_candidates, 
				SurvivorType *ptr,
				int num_bits,
				BitBuffer *out_buffer);
  TTPathSegment * TTMaxSegOverlap(int num);

  int TTSearch(int candidate_num);

  void TTAddSegStack(TTStack *pathStack, int stackSize, 
		     TTPathSegment *seg,
		     float metric);
  void FreeTTPathSegment(TTPathSegment *seg);

  int GetMemSize();

  void configure_parity_info(int period, int pstart);

  int set_rate_sched(RateSchedType *in_rate_sched);
  

};  // end of class < RCPCDecoder >



// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

//  Inline functions

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =


// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// end of file < rcpc.h >

#endif // __RCPC_H
