// -*- c++ -*-
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

//       R C P C   ( E N / D E ) C O D E R   C L A S S E S

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
//
// P. Greg Sherwood            sherwood@code.ucsd.edu
// Kenneth Zeger	       zeger@code.ucsd.edu
//
// 9500 Gilman Dr MC 0407
// La Jolla, CA 92093
//
//
// Copyright (c) 1997 P. Greg Sherwood and Kenneth Zeger
//
// This program is Copyright (c) by P. Greg Sherwood and Kenneth Zeger
// It may not be redistributed without written permission of its 
// copyright holders. It may not be sold for profit or
// incorporated in commercial programs without the written permission
// of the copyright holders. This program is provided as is, without any
// express or implied warranty, without even the warranty of fitness
// for a particular purpose.
//


// - - Inclusion - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#include "rcpc.h"

#define DEBUG 0

/**************************************************
 * The following codes and puncturing matrices 
 * are from:
 * J. Hagenauer "Rate Compatible Convolutional Codes 
 *     and their Applications," IEEE Trans. Comm., vol. 36
 *     No. 4, April 1988, pp. 389-400.
 **************************************************/
#define CODE 3
#if (CODE == 1)
static int GenMat[] = {0x13, 0x1D, 0x17, 0x1B};
static int BaseNout = 4;
static int MemSize = 4;
static int Period = 8;
static int PunctMat[] = { 
  0xC, 0x8, 0x8, 0x8, 0x4, 0x8, 0x8, 0x8,
  0xC, 0x8, 0x8, 0x8, 0xC, 0x8, 0x8, 0x8,
  0xC, 0x8, 0xC, 0x8, 0xC, 0x8, 0xC, 0x8,
  0xC, 0xC, 0xC, 0x8, 0xC, 0xC, 0xC, 0x8,
  0xC, 0xC, 0xC, 0xC, 0xC, 0xC, 0xC, 0xC,
  0xE, 0xC, 0xC, 0xC, 0xE, 0xC, 0xC, 0xC,
  0xE, 0xE, 0xC, 0xC, 0xE, 0xE, 0xC, 0xC,
  0xE, 0xE, 0xE, 0xC, 0xE, 0xE, 0xE, 0xC,
  0xE, 0xE, 0xE, 0xE, 0xE, 0xE, 0xE, 0xE,
  0xF, 0xE, 0xE, 0xE, 0xF, 0xE, 0xE, 0xE,
  0xF, 0xE, 0xF, 0xE, 0xF, 0xE, 0xF, 0xE,
  0xF, 0xF, 0xF, 0xE, 0xF, 0xF, 0xF, 0xE,
  0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF, 0xF
};
static int NumCodes = (sizeof(PunctMat)/sizeof(PunctMat[0]))/Period;

#endif

#if (CODE == 2)
static int GenMat[] = {0x3D, 0x2B, 0x27};
static int BaseNout = 3;
static int MemSize = 5;
static int Period = 8;
static int PunctMat[] = { 
  0x6, 0x4, 0x4, 0x4, 0x4, 0x4, 0x4, 0x4,
  0x6, 0x4, 0x4, 0x4, 0x6, 0x4, 0x4, 0x4,
  0x6, 0x4, 0x6, 0x4, 0x6, 0x4, 0x6, 0x4,
  0x6, 0x6, 0x6, 0x4, 0x6, 0x6, 0x6, 0x4,
  0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6,
  0x7, 0x6, 0x6, 0x6, 0x7, 0x6, 0x6, 0x6,
  0x7, 0x6, 0x7, 0x6, 0x7, 0x6, 0x7, 0x6,
  0x7, 0x7, 0x7, 0x6, 0x7, 0x7, 0x7, 0x6,
  0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7,
};
static int NumCodes = (sizeof(PunctMat)/sizeof(PunctMat[0]))/Period;
#endif


#if (CODE==3)

static int GenMat[] = {0x5B, 0x79, 0x65};
static int BaseNout = 3;
static int MemSize = 6;
static int Period = 8;
static int PunctMat[] = { 
  0x6, 0x4, 0x4, 0x4, 0x4, 0x4, 0x4, 0x4,
  0x6, 0x4, 0x4, 0x4, 0x6, 0x4, 0x4, 0x4,
  0x6, 0x4, 0x6, 0x4, 0x6, 0x4, 0x6, 0x4,
  0x6, 0x6, 0x6, 0x4, 0x6, 0x6, 0x6, 0x4,
  0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6,
  0x7, 0x6, 0x6, 0x6, 0x7, 0x6, 0x6, 0x6,
  0x7, 0x7, 0x6, 0x6, 0x7, 0x7, 0x6, 0x6,
  0x7, 0x7, 0x7, 0x6, 0x7, 0x7, 0x7, 0x6,
  0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7,
};
static int NumCodes = (sizeof(PunctMat)/sizeof(PunctMat[0]))/Period;

#endif

// - - External definitions  - - - - - - - - - - - - - - - - - - - - - -


// - - Member Functions  - - - - - - - - - - - - - - - - - - - - - - - -
RCPCEncoder::RCPCEncoder(RCPCCodeSpec *in_code_spec, RateSchedType *in_rate_sched, int code_number)
{
  rcpc_buffer = 0;
  rcpc_mem_full = 0;
  loc = 0;
  sched_idx = -1;
  rate_sched = NULL;

  if (in_code_spec == NULL)
    { // set to default values
      code_spec.BaseNout = BaseNout;
      code_spec.MemSize = MemSize;
      code_spec.Period = Period;
      code_spec.NumCodes = NumCodes;
      code_spec.GenMat = GenMat;
      code_spec.PunctMat = PunctMat;

    }
  else
    { // copy values to structure
      code_spec = *in_code_spec;
      // allocate space for GenMatrix and PunctMatrix
      NEW_ARRAY(code_spec.GenMat, code_spec.BaseNout, int);
      NEW_ARRAY(code_spec.PunctMat, code_spec.NumCodes*code_spec.Period,
		 int);
      // copy values 
      memcpy(code_spec.GenMat, in_code_spec->GenMat, code_spec.BaseNout*sizeof(int));
      memcpy(code_spec.PunctMat, in_code_spec->PunctMat, 
	     code_spec.NumCodes*code_spec.Period*sizeof(int));
    }

  code_num = code_number;
  if (code_num < 0 || code_num >= code_spec.NumCodes)
    code_num = 0;
      

  if (in_rate_sched && in_rate_sched->size > 0)
    {
      NEW_OBJECT(rate_sched, RateSchedType);
      rate_sched->size = in_rate_sched->size;

      // copy the values 
      NEW_ARRAY(rate_sched->sched_array, rate_sched->size, 
		 SchedEntryType);

      // copy the values
      memcpy(rate_sched->sched_array, in_rate_sched->sched_array, 
	     rate_sched->size*sizeof(SchedEntryType));
#if DEBUG
  printf("RCPCEncoder: Rate schedule of size %d specified.  first two pairs are: (%d, %d) (%d, %d)\n",
	 rate_sched->size, rate_sched->sched_array[0].location,rate_sched->sched_array[0].code_num, rate_sched->sched_array[1].location,rate_sched->sched_array[1].code_num);
#endif


      if (rate_sched->sched_array[0].location == 0)
	{ // set first code_num
	  code_num = rate_sched->sched_array[0].code_num;
	  sched_idx = 1;
	}
      else
	{
	  sched_idx = 0;
	}
  
    }

#if DEBUG
  printf("RCPCEncoder: Code number %d, period %d, num out %d mem %d\n",
	 code_num, code_spec.Period, code_spec.BaseNout, code_spec.MemSize);
#endif

  // points to current code's Puncture matrix.
  punc_ptr = code_spec.PunctMat + (code_num*code_spec.Period);

  per_idx = 0;
  oindex = (1 << (code_spec.BaseNout-1));
  punc_entry = punc_ptr[per_idx];

}

RCPCEncoder::RCPCEncoder(RateSchedType *in_rate_sched, int code_number)
{
  RCPCEncoder(NULL, in_rate_sched,  code_number);
}


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// Destructor
RCPCEncoder::~RCPCEncoder()
{
  if (code_spec.GenMat != GenMat)
    DELETE_ARRAY(code_spec.GenMat);
  if (code_spec.PunctMat != PunctMat)
    DELETE_ARRAY(code_spec.PunctMat);

  if (rate_sched)
    {
      if (rate_sched->sched_array)
	{
	  DELETE_ARRAY(rate_sched->sched_array);
	}
      DELETE_OBJECT(rate_sched);
    }

}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void RCPCEncoder::reset(int state, int punc_phase)
{
  // Does not currently handle rate schedules 

  rcpc_buffer = state;
  punc_ptr = code_spec.PunctMat + (code_num*code_spec.Period);
  per_idx = (punc_phase % code_spec.Period);
  oindex = (1 << (code_spec.BaseNout-1));
  punc_entry = punc_ptr[per_idx];
  loc = 0;

}

void RCPCEncoder::set_state(int state)
{
  reset();
  rcpc_buffer = state;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
int RCPCEncoder::compute_coded_bits(int raw_bits, int code_num)
{
  int ii;
  int bits_per_period;
  int num_periods;
  int total_bits;
  int *punc_ptr;

  if (code_num < 0 || code_num >= code_spec.NumCodes)
    {
      ErrorMsg("RCPCEncoder", "compute_coded_bits", "code_number %d out of range [%d, %d]\n", code_num, 0, code_spec.NumCodes-1);
      return -1;
    }


  num_periods = raw_bits/code_spec.Period;

  punc_ptr = code_spec.PunctMat + (code_num*code_spec.Period);
  for (bits_per_period = 0, ii = 0; ii < code_spec.Period; ++ii)
    {
      bits_per_period += weight(punc_ptr[ii]);
    }

  // printf("Computed bits_per_period = %d for code %d\n", bits_per_period, code_num);

  total_bits = bits_per_period*num_periods;

  //  printf("Bits based on integer periods (%d) = %d\n", num_periods, total_bits);

  // get the remaining bits
  for (ii = 0; ii < (raw_bits % code_spec.Period); ++ii)
    {
      total_bits += weight(punc_ptr[ii]);
    }

  // printf("Bits after remainder (%d) = %d\n",(raw_bits % code_spec.Period), total_bits);
  
  return total_bits;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
int RCPCEncoder::compute_coded_bits(int raw_bits)
{
  return compute_coded_bits(raw_bits, code_num);
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
int RCPCEncoder::compute_src_bits(int coded_bits, int code_num)
{
  int ii;
  int raw_bits;
  int *punc_ptr;

  if (code_num < 0 || code_num >= code_spec.NumCodes)
    {
      ErrorMsg("RCPCEncoder", "compute_src_bits", "code_number %d out of range [%d, %d]\n", code_num, 0, code_spec.NumCodes-1);
      return -1;
    }
  punc_ptr = code_spec.PunctMat + (code_num*code_spec.Period);
  for (raw_bits = 0, ii = 0; coded_bits > 0; )
    {
      coded_bits -= weight(punc_ptr[ii]);
      if (coded_bits >= 0)
	++raw_bits;
      ii = (ii + 1) % code_spec.Period;
    }

  return raw_bits;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
int RCPCEncoder::compute_src_bits(int coded_bits)
{

  return compute_src_bits(coded_bits, code_num);
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int RCPCEncoder::send_bit(void )
{

  if (!oindex)
    {
      /* adjust per_idx */
      per_idx = (per_idx + 1) % code_spec.Period;
      punc_entry = punc_ptr[per_idx];
      oindex = (1 << (code_spec.BaseNout-1));
    }

  if (punc_entry & oindex)
    {	/* bit is NOT punctured so output */
      oindex >>= 1;
      return 1;
    }

  else
    {
      oindex >>= 1;
      return 0;
    }

}


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

void RCPCEncoder::encode(int bits, int *input, BitBuffer *buffer)
{
  int ii, jj, kk, num_ints;
  unsigned int mask;
  int outb;
  char y;

  num_ints = bits/32;

  rcpc_mem_full = 1;

  for (ii = 0; ii < num_ints; ++ii)
    {
      for (jj = 0, mask = 0x80000000 ; jj < 32; ++jj, mask >>= 1)
	{
	  rcpc_buffer <<= 1;
	  if (mask & input[ii]) rcpc_buffer |= 1;
	  --bits;
	  
	  /* now compute the output bit */
	  for (kk = 0; kk < code_spec.BaseNout; ++kk)
	    {
	      if (send_bit())
		{
		  outb = rcpc_buffer & code_spec.GenMat[kk];
		  /* now compute the parity of outb */
		  for (y = 0; outb ; outb >>= 1)
		    y ^= (outb & 1);
		  buffer->buffer_bit(y);
		}
	    }
	}

    }

  for (ii = num_ints, mask = (1 << (bits-1)); bits; --bits, mask >>= 1)
    {
      rcpc_buffer <<= 1;
      if (mask & input[ii]) rcpc_buffer |= 1;

      
      /* now compute the output bit */
      for (kk = 0; kk < code_spec.BaseNout; ++kk)
	{
	  if (send_bit())
	    {
	      outb = rcpc_buffer & code_spec.GenMat[kk];
	      /* now compute the parity of outb */
	      for (y = 0; outb ; outb >>= 1)
		y ^= (outb & 1);
	      buffer->buffer_bit(y);
	    }
	  
	}

    }

}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int RCPCEncoder::encode(int bits, BitBuffer *input, BitBuffer *buffer)
{
  int ii, jj, kk, bcnt, num_words;
  int in_word;
  unsigned int mask;
  int outb;
  char y;
  int start_cnt;

  // get the number of bits in the output buffer
  start_cnt = buffer->bits_in_buffer();
  bits = Max(Min(bits, input->rbits_to_end()), 0);
  num_words = bits/16;
  rcpc_mem_full = 1;

  for (ii = 0; ii < num_words; ++ii)
    {
      if ((bcnt = input->read_bits(16, in_word)) <= 0)
	break;
      for (jj = 0, mask = (1 << (bcnt - 1)); jj < bcnt; ++jj, mask >>= 1)
	{
	  rcpc_buffer <<= 1;
	  if (mask & in_word) rcpc_buffer |= 1;
	  --bits;

	  for (kk = 0; kk < code_spec.BaseNout; ++kk)
	    {
	      if (send_bit())
		{
		  outb = rcpc_buffer & code_spec.GenMat[kk];
		  /* now compute the parity of outb */
		  for (y = 0; outb ; outb >>= 1)
		    y ^= (outb & 1);
		  buffer->buffer_bit(y);
		  ++loc;
		}
	    }
	  // check if location exceeds sched spec
	  if (rate_sched && sched_idx >= 0 && 
	      (loc >= rate_sched->sched_array[sched_idx].location))
	    {
	      code_num = rate_sched->sched_array[sched_idx].code_num;
	      punc_ptr = code_spec.PunctMat + (code_num*code_spec.Period);
	      // Set sched_idx to -1 if at the last code so we no longer
	      // worry about code changes
#if DEBUG
	      printf("RCPCEncoder: changed code num to %d at loc %d\n",
		     code_num, loc);
#endif
	      sched_idx = (sched_idx < rate_sched->size - 1) ? sched_idx+1 : -1;
	      
	    }
	}
    }


  for ( ; bits; --bits)
    {
      if ((in_word = input->read_bits(1)) < 0)
	break;
      rcpc_buffer <<= 1;
      if (in_word) rcpc_buffer |= 1;
      
      for (kk = 0; kk < code_spec.BaseNout; ++kk)
	{
	  if (send_bit())
	    {
	      outb = rcpc_buffer & code_spec.GenMat[kk];
	      /* now compute the parity of outb */
	      for (y = 0; outb ; outb >>= 1)
		y ^= (outb & 1);
	      buffer->buffer_bit(y);
	      ++loc;
	    }
	}
      // check if location exceeds sched spec
      if (rate_sched && sched_idx >= 0 && 
	  (loc >= rate_sched->sched_array[sched_idx].location))
	{
	  code_num = rate_sched->sched_array[sched_idx].code_num;
	  punc_ptr = code_spec.PunctMat + (code_num*code_spec.Period);
	  // Set sched_idx to -1 if at the last code so we no longer
	  // worry about code changes
#if DEBUG
	  printf("RCPCEncoder: changed code (ii) num to %d at loc %d\n",
		 code_num, loc);
#endif
	  sched_idx = (sched_idx < rate_sched->size - 1) ? sched_idx+1 : -1;
	  
	}
    }
	  
  bits = buffer->bits_in_buffer() - start_cnt;

  return bits;
}
  

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int RCPCEncoder::max_code_index(void)
{
  return (code_spec.NumCodes-1);
}


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

void RCPCEncoder::select_code(int code_number)
{
  if (code_number < 0 || code_number >= code_spec.NumCodes)
    {
      ErrorMsg("RCPCEncoder", "select_code", "code_number %d out of range [0, %d]", code_number, code_spec.NumCodes-1);
      return;
    }
  code_num = code_number;
  punc_ptr = code_spec.PunctMat + (code_num*code_spec.Period);
  if (oindex == (1 << (code_spec.BaseNout - 1)))
    punc_entry = punc_ptr[per_idx];
      
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int RCPCEncoder::flush_rcpc_mem(BitBuffer *buffer)
{
  int ii, kk, y;
  int outb;
  int bcnt = 0;

  if (rcpc_mem_full)
    {
      /* flush the encoder */
      for (ii = 0; ii < code_spec.MemSize; ++ii)
	{
	  rcpc_buffer <<= 1;
	  /* now compute the output bit */
	  for (kk = 0; kk < code_spec.BaseNout; ++kk)
	    {
	      if (send_bit())
		{
		  outb = rcpc_buffer & code_spec.GenMat[kk];
		  /* now compute the parity of outb */
		  for (y = 0; outb ; outb >>= 1)
		    y ^= (outb & 1);
		  buffer->buffer_bit(y);
		  ++bcnt;
		  ++loc;
		}
	    }

	  // check if location exceeds sched spec
	  if (rate_sched && sched_idx >= 0 && 
	      (loc >= rate_sched->sched_array[sched_idx].location))
	    {
	      code_num = rate_sched->sched_array[sched_idx].code_num;
	      punc_ptr = code_spec.PunctMat + (code_num*code_spec.Period);
	      // Set sched_idx to -1 if at the last code so we no longer
	      // worry about code changes
#if DEBUG
	      printf("RCPCEncoder: changed code (iii) num to %d at loc %d\n",
		     code_num, loc);
#endif
	      sched_idx = (sched_idx < rate_sched->size - 1) ? sched_idx+1 : -1;
	      
	    }
	  
	}
    }
  rcpc_mem_full = 0;
  return bcnt;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int RCPCEncoder::GetMemSize()
{
  return code_spec.MemSize;
}





// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

//  Member functions of the class < RCPCDecoder >

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =


// - - Constructor - - - - - - - - - - - - - - - - - - - - - - - - - - -
RCPCDecoder::RCPCDecoder(RCPCCodeSpec *in_code_spec,
			 RateSchedType *in_rate_sched,
			 int code_number, int trellis_stages,
			 int tree_trellis_mode, 
			 int buf_size, int seg_size)
{
  int ii, jj, kk;
  int state, out, y;

  if (in_code_spec == NULL)
    { // set to default values
      code_spec.BaseNout = BaseNout;
      code_spec.MemSize = MemSize;
      code_spec.Period = Period;
      code_spec.NumCodes = NumCodes;
      code_spec.GenMat = GenMat;
      code_spec.PunctMat = PunctMat;
    }
  else
    { // copy values to structure
      code_spec = *in_code_spec;
      // allocate space for GenMatrix and PunctMatrix
      NEW_ARRAY(code_spec.GenMat, code_spec.BaseNout, int);

      NEW_ARRAY(code_spec.PunctMat, code_spec.NumCodes*code_spec.Period,
		 int);
      // copy values 
      memcpy(code_spec.GenMat, in_code_spec->GenMat, code_spec.BaseNout*sizeof(int));
      memcpy(code_spec.PunctMat, in_code_spec->PunctMat, 
	     code_spec.NumCodes*code_spec.Period*sizeof(int));
    }


  /* compute the number of states */
  num_states = 1 << code_spec.MemSize;
  code_num = code_number;
  if (code_num < 0 || code_num >= code_spec.NumCodes)
    code_num = 0;
  loc = 0;
  sched_idx = -1;
  rate_sched = NULL;

  if (in_rate_sched && in_rate_sched->size > 0)
    {
      NEW_OBJECT(rate_sched, RateSchedType);
      rate_sched->size = in_rate_sched->size;

      // copy the values 
      NEW_ARRAY(rate_sched->sched_array, rate_sched->size, 
		 SchedEntryType);

      // copy the values
      memcpy(rate_sched->sched_array, in_rate_sched->sched_array, 
	     rate_sched->size*sizeof(SchedEntryType));

      if (rate_sched->sched_array[0].location == 0)
	{ // set first code_num
	  code_num = rate_sched->sched_array[0].code_num;
	  sched_idx = 1;
	}
      else
	{
	  sched_idx = 0;
	}

#if DEBUG
      printf("RCPCDecoder constructor: init rate schedule of size %d code_num=%d next transition = %d\n", rate_sched->size, code_num, rate_sched->sched_array[sched_idx]);
#endif
 
    }


  num_stages = trellis_stages;
  tt_mode = (tree_trellis_mode != 0);

  tt_stack_size = 0;
  tt_stack = NULL;

  parity_array = working_parity_array = NULL;
  inner_phase = parity_start = inner_period = 0;


  /* create the state mask */
  for (ii = 0, state_mask = 0; ii < code_spec.MemSize; 
       ++ii, state_mask = (state_mask << 1) | 1);

#if DEBUG
  printf("Rate %d, memory %d, period %d, num_states %d, state_mask %X\n",
	 code_spec.BaseNout, code_spec.MemSize, code_spec.Period, num_states, state_mask);
#endif

  punc_ptr = code_spec.PunctMat + (code_num*code_spec.Period);

#if DEBUG
  for (ii = 0; ii < code_spec.BaseNout; ++ii)
    {
      printf("0x%X\n", code_spec.GenMat[ii]);
    }
  for (ii = 0; ii < code_spec.NumCodes; ++ii)
    {
      for (jj = 0; jj < code_spec.Period; ++jj)
	{
	  printf("0x%X ", code_spec.PunctMat[ii*code_spec.Period + jj]);
	}
      printf("\n");
    }
#endif

  stage_size = (tt_mode + 1)*num_states;
  
  path_buffer = NULL;
  pathbuf_size = buf_size;
  pathseg_size = seg_size;
      
  
  // allocate tmp working space for a trellis stage
  NEW_ARRAY(working_trellis_stage, stage_size,
	     SurvivorType);



  NEW_ARRAY(token_buffer, num_states, int);

  NEW_ARRAY(new_token, num_states, int);




  /* allocate enough space for the branch labels */
  NEW_ARRAY(labels, (num_states << 1), int);

  reset_trellis();
  
  
  /* generate the array of branch labels for the base code for 
   * all state transitions.  
   */

  for (ii = 0; ii < num_states; ++ii)
    {
      for (jj = 0; jj < 2; ++jj)
	{
	  state = ((ii << 1) | jj) & state_mask;
	  for (labels[2*ii+jj] = 0, kk = 0; 
	       kk < code_spec.BaseNout;
	       ++kk)
	    {
	      out = ((ii << 1) | jj) & code_spec.GenMat[kk];
	      /* get the parity of out */
	      for (y = 0; out; out >>=1)
		y ^= (out & 1);
	      labels[2*ii + jj] = (labels[2*ii + jj] << 1) | y;
				   
	    }
	}
    }

  reset_memory();
  clear_tt_stack();

#if DEBUG
  printf("RCPCDecoder constructor: bits_for_transition %d code_num=%d",
	 bits_for_transition, code_num);
  if (rate_sched)
    printf(" next transition = %d", rate_sched->sched_array[sched_idx]);
  printf("\n");
#endif


}

RCPCDecoder::RCPCDecoder(RateSchedType *in_rate_sched,
			 int code_number, int trellis_stages,
			 int tree_trellis_mode, 
			 int buf_size, int seg_size)
{
  RCPCDecoder(NULL, 
	      in_rate_sched,
	      code_number,  trellis_stages,
	      tree_trellis_mode, 
	      buf_size,  seg_size);
}


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// - - Destructor  - - - - - - - - - - - - - - - - - - - - - - - - - - -
RCPCDecoder::~RCPCDecoder()
{

  if (code_spec.GenMat != GenMat)
    DELETE_ARRAY(code_spec.GenMat);
  if (code_spec.PunctMat != PunctMat)
    DELETE_ARRAY(code_spec.PunctMat);

  if (rate_sched)
    {
      if (rate_sched->sched_array)
	{
	  DELETE_ARRAY(rate_sched->sched_array);
	}
      DELETE_OBJECT(rate_sched);
    }

  if (labels)
    {
      DELETE_ARRAY(labels);
    }
  if (token_buffer)
    {
      DELETE_ARRAY(token_buffer);
    }
  if (new_token)
    {
      DELETE_ARRAY(new_token);
    }

  if (working_trellis_stage)
    {
      DELETE_ARRAY(working_trellis_stage);
    }


  clear_trellis();

  clear_tt_stack();

  if (parity_array)
    {
      DELETE_ARRAY(parity_array);
    }

  if (working_parity_array)
    {
      DELETE_ARRAY(working_parity_array);
    }

}


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int RCPCDecoder::GetMemSize()
{
  return code_spec.MemSize;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

void RCPCDecoder::recurse_buffer(SurvivorType *ptr, int level, 
				 BitBuffer *out_buffer)
{
  if (level)
    recurse_buffer(ptr->backptr, level-1, out_buffer);
#if DEBUG
  printf("Metric is %f, state = %d\n", ptr->metric, ptr->state);
#endif
  out_buffer->buffer_bit(ptr->state & 1);
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void RCPCDecoder::configure_parity_info(int period, int pstart)
{
  inner_period = (period > 0) ? period : 0;
  parity_start = (pstart > 0 && pstart < period) ? pstart : period;
  inner_phase = 0;
}


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

void RCPCDecoder::reset_trellis()
{
  int ii, jj;
  SurvivorType *ptr;


  if (path_buffer != NULL)
    clear_trellis();

  reset_memory();


  if (num_stages <= 0)
    { // don't do truncated memory decoding
      num_stages = 0;
      
      
      NEW_ARRAY(path_buffer, pathbuf_size, SurvivorType **);

      memset(path_buffer, 0, pathbuf_size*sizeof(SurvivorType **));

      // allocate the first array for stage ptrs
      NEW_ARRAY(path_buffer[0], pathseg_size,
		 SurvivorType *);

      memset(path_buffer[0], 0, pathseg_size*sizeof(SurvivorType *));

      NEW_ARRAY(path_buffer[0][0], stage_size, SurvivorType);

      for (ptr = path_buffer[0][0], jj = 0; jj < stage_size; ++jj, ++ptr)
	{
	  ptr->state = jj; ptr->metric = 0; ptr->backptr = NULL;
	}
      
      
    }
  else
    { // will only use num_stages SurvivorType arrays
      NEW_ARRAY(path_buffer, 1, SurvivorType **);

      NEW_ARRAY(path_buffer[0], num_stages,
		 SurvivorType *);

      // allocate the SurvivorType Arrays
      for (ii = 0; ii < num_stages; ++ii)
	{
	  
	  NEW_ARRAY(path_buffer[0][ii], stage_size, SurvivorType);

	  for (ptr = path_buffer[0][ii], jj = 0; jj < stage_size; ++jj, ++ptr)
	    {
	      ptr->state = jj; ptr->metric = 0; ptr->backptr = NULL;
	    }
	  
	}

    }
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


void RCPCDecoder::reset_memory()
{
  if (token_buffer)
    {
      memset(token_buffer, 0, num_states*sizeof(int));
      token_buffer[0] = 1;
    }

  pathbuf_idx = pathseg_idx = 0;

  /* set the Period index and output index to 0 */
  per_idx = 0;
  
  
  
  /* build the relevant puncture entry */
  punc_entry = punc_ptr[per_idx];
  /* determine the number of bits to get for next 
   * state transition */
  bits_for_transition = weight(punc_entry);

  current = 0;
  buffer_full = 0;


  
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

void RCPCDecoder::clear_trellis()
{
  int ii,jj;

  if (path_buffer)
    {
      if (num_stages)
	{
	  if (path_buffer[0] != NULL)
	    {

	      for (ii = 0; ii < num_stages; ++ii)
		{
		  if (path_buffer[0][ii] != NULL)
		    {
		      DELETE_ARRAY(path_buffer[0][ii]);
		    }
		}
	      DELETE_ARRAY(path_buffer[0]);
	    }
	  DELETE_ARRAY(path_buffer);
	}
      else
	{
	  for (ii = 0; ii < pathbuf_idx; ++ii)
	    {
	      for (jj = 0; jj < pathseg_size;++jj)
		{
		  if (path_buffer[ii][jj] != NULL)
		    {
		      // cout << "deleting path buffer " << ii << ", " << jj << endl;
		      DELETE_ARRAY(path_buffer[ii][jj]);
		    }
		}
	      if (path_buffer[ii] != NULL)
		{
		  // cout << "deleting path buffer " << ii <<  endl;
		  DELETE_ARRAY(path_buffer[ii]);
		}
	    }
	  if (path_buffer[pathbuf_idx] != NULL)
	    {
	      for (jj = 0; jj <= pathseg_idx; ++jj)
		{
		  if (path_buffer[pathbuf_idx][jj] != NULL)
		    {
		      DELETE_ARRAY(path_buffer[pathbuf_idx][jj]);
		    }
		}
	      DELETE_ARRAY(path_buffer[pathbuf_idx]);
	    }
	  DELETE_ARRAY(path_buffer);
	}

    }
  pathbuf_idx = 0;
  pathseg_idx = 0;
  buffer_full = 0;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


void RCPCDecoder::clear_tt_stack()
{
  int ii;

  if (tt_stack_size != 0 && tt_stack != NULL)
    {
      for (ii = 0; ii < tt_stack_size; ++ii)
	{
	  FreeTTPathSegment(tt_stack[ii].path);
	}
      DELETE_ARRAY(tt_stack);
    }
     
  tt_last_path_found = tt_stack_size = 0; tt_stack = NULL;
  

}


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

void RCPCDecoder::terminate_trellis(BitBuffer *out_buffer, int num_candidates)
{
  /* flush out the decision buffer.
   * Assume the trellis is properly terminated to the
   * all zero state 
   */
  int ii;
  int buffer_size;
  SurvivorType *ptr, *current_stage;

  if (num_stages)
    {
      current_stage = path_buffer[0][pathseg_idx];
      buffer_size = num_stages;
    }
  else
    {
      current_stage = path_buffer[pathbuf_idx][pathseg_idx];
      buffer_size = (pathbuf_idx*pathseg_size) + pathseg_idx +1;
#if DEBUG
      printf("Within terminate_trellis - buffer size = %d, buf_idx = %d, seg_idx = %d\n", buffer_size, pathbuf_idx, pathseg_idx);
#endif
    }

#if DEBUG
  for (ii = 0; ii < num_states; ++ii)
    {
      printf("Metric for state %d is %f\n", ii, 
	     current_stage[ii].metric);
    }
#endif

  /* skip last memory transitions which are flush bits */
  //for (ptr = current_stage, ii = 0;
  //ii < code_spec.MemSize; ++ii)
  //ptr = ptr->backptr;
  ptr = current_stage;

  if (tt_mode && num_candidates > 1)
    {
      find_multiple_candidates(num_candidates, 
			       ptr, 
			       //buffer_size - code_spec.MemSize - 1,
			       buffer_size-1, out_buffer);
    }
  else if (num_stages > 0 && num_stages < 100)
    {
      /* skip last memory transitions which are flush bits */
      for (ptr = current_stage, ii = 0;ii < code_spec.MemSize; ++ii)
	ptr = ptr->backptr;

      recurse_buffer(ptr, 
		     buffer_size - code_spec.MemSize -2 , 
		     out_buffer); // old num was buffer_size-2
    }
  else
    {
      TTPathSegment tseg;

      tseg.start = ptr;
      //tseg.length = buffer_size - code_spec.MemSize - 1;
      tseg.length = buffer_size-1;
      tseg.next = NULL;
      
      decode_TTpath(&tseg, out_buffer);

    }
  memset(token_buffer, 0, num_states*sizeof(int));
  token_buffer[0] = 1;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

void RCPCDecoder::init_tt_stack(int num_elements)
{
  int ii;

  if (tt_stack == NULL)
    {
      NEW_ARRAY(tt_stack, num_elements, TTStack);
      tt_stack_size = num_elements;
    }
  else if (num_elements > tt_stack_size)
    {
      clear_tt_stack();
      NEW_ARRAY(tt_stack, num_elements, TTStack);

      tt_stack_size = num_elements;
    }
  // initialize the values
  for (ii = 0; ii < num_elements; ++ii)
    {
      tt_stack[ii] = TTStack(INIT_METRIC, NULL);
    }

}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int RCPCDecoder::decode_nth_candidate(int num, BitBuffer *out_buffer)
{
  if (tt_stack == NULL || num < 0 || num >= tt_stack_size)
    return -1;

#if DEBUG
  printf("RCPCDecoder: decoding %d candidate with metric %f\n",
	 num-1, tt_stack[num-1].metric);
#endif

  // search for path if necessary
  while (tt_last_path_found < num )
    {
      if (!TTSearch(tt_last_path_found+1))
	return -1;
    }
  
  decode_TTpath(tt_stack[num].path, out_buffer);

  return 0;
}


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

void RCPCDecoder::find_multiple_candidates(int num_candidates, 
					   SurvivorType *ptr,
					   int num_bits,
					   BitBuffer *out_buffer)
{
  TTPathSegment *tSeg;

  // initialize to stack to max number of candidates
  init_tt_stack(num_candidates);

  // put the best path at the top of the stack
  tt_stack[0].metric = ptr->metric;
  // Create new path segment
  NEW_OBJECT(tSeg, TTPathSegment);
  tt_stack[0].path = tSeg;
  tt_stack[0].path->start = ptr;
  tt_stack[0].path->length = num_bits;
  tt_stack[0].path->next = NULL;

  tt_last_path_found = 0;

#if DEBUG
  printf("FindMultCandidates: top of stack metric = %f\n", tt_stack[0].metric);
#endif

  // TTSearch(num_candidates); // old call when all candidates were found up front
  
  // after finding candidate paths - decode the most likely
#if DEBUG
  printf("FindMultCandidates: now decoding path 0 of length %d\n",
	 num_bits);
  for (cnum = 0; cnum < num_candidates; ++cnum)
    {
      printf("FindMultCandidates: metric for path %d is %f\n",
	     cnum, tt_stack[cnum].metric);
    }
#endif

  // now just decode the best candidate
  decode_TTpath(tt_stack[0].path, out_buffer, num_bits);


}


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

TTPathSegment *RCPCDecoder::TTMaxSegOverlap(int num)
{
  /* Find the 1st segment that deviates from the a higher 
   * path with maximum overlap 
   */
  
  TTPathSegment *seg;
  TTPathSegment *pSeg;
  TTPathSegment *maxSeg = tt_stack[num].path;
  int maxCnt = -1;
  int pnum = 0;
  int cnt;
  int ii;



  for (ii = 0; ii < num; ++ii)
    { /* loop over higher paths */
      pSeg = tt_stack[ii].path;
      cnt = 0;
      for (seg = tt_stack[num].path; seg != NULL; seg = seg->next)
	{
	  if (pSeg == NULL)
	    break;
	  if (pSeg->start != seg->start)
	    break;
	  if ( pSeg->length != seg->length)
	    {
	      cnt += Min(pSeg->length, seg->length);
	      seg = seg->next;
	      break;
	    }
	  cnt+=pSeg->length;
	  pSeg = pSeg->next;

	}
      if (cnt > maxCnt)
	{
	  pnum = ii;
	  maxCnt = cnt;
	  maxSeg = seg;
	}
    }

  return maxSeg;
      
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int RCPCDecoder::TTSearch(int candidate_num)
{
  int ii;
  int scnt;
  TTPathSegment *startSeg;
  TTPathSegment *curSeg;
  TTPathSegment *seg, *tSeg, *lastSeg, *newSeg;
  SurvivorType *sptr;
  SurvivorType *p;


  float metric;
  float *stackMax;

  if (candidate_num <= 0 || candidate_num >= tt_stack_size)
    return 0;

  stackMax = &(tt_stack[tt_stack_size - 1].metric);

  // path to expand is previous
  ii = candidate_num - 1;


  if (tt_stack[ii].path == NULL)
    {
      ErrorMsg("RCPCDecoder","TTSearch", 
	       "Unable to decode candidate %d -- stopped at %d\n",
	       candidate_num, ii);
      return 0;
    }

  startSeg = (ii==0) ? tt_stack[0].path : TTMaxSegOverlap(ii);
  
  /* traverse all segments of the current path */
  for (curSeg = startSeg; curSeg != NULL; curSeg = curSeg->next)
    {
      /* Step back through each stage in this segment */
      for (sptr = curSeg->start, scnt = 0; 
	   sptr != NULL && scnt < (curSeg->length - 1); 
	   sptr = sptr->backptr, ++scnt)
	{
	  /* check other path */
	  p = sptr + 1;
	  metric = p->metric + (tt_stack[ii].metric - sptr->metric);
	  
	  
#if (1)
	  if (metric < tt_stack[ii].metric)
	    {
	      ErrorMsg("RCPCDecoder", "TTSearch", 
		       "Unexpected metric O:%f < Tok %f\n ii=%d, scnt = %d\n",
		      metric, tt_stack[ii].metric,ii,scnt);
	    }
#endif
	  if (metric < *stackMax)
	    { /* Create path segments */
	      
#if (DEBUG > 10)
	      if (p->backptr)
		printf("TTSearch: ii=%d, stack metric = %f, sptr metric %f, p metric %f, backptr 0x%X p state %d b state %d\n",  ii, tt_stack[ii].metric, sptr->metric, p->metric, p->backptr, p->state, p->backptr->state);
	      else
		printf("TTSearch: ii=%d, stack metric = %f, sptr metric %f, p metric %f, backptr 0x%X p state %d\n",  ii, tt_stack[ii].metric, sptr->metric, p->metric, p->backptr, p->state);
	      
#endif

	      newSeg = lastSeg = NULL;
	      /* Copy overlap region */
	      for (seg = tt_stack[ii].path; seg != curSeg; 
		   seg = seg->next)
		{
		  NEW_OBJECT(tSeg, TTPathSegment);

		  *tSeg = *seg;
		  if (newSeg == NULL)
		    {
		      newSeg = tSeg;
		      lastSeg = tSeg;
		    }
		  else
		    {
		      lastSeg->next = tSeg;
		      lastSeg = tSeg;
		    }
		}
	      NEW_OBJECT(tSeg, TTPathSegment);

	      *tSeg = *curSeg;
	      if (newSeg == NULL)
		newSeg = tSeg;
	      else
		lastSeg->next = tSeg;
	      tSeg->next = NULL;
	      tSeg->length = scnt + 1; // changed 2 -> 1
	      lastSeg = tSeg;
	      NEW_OBJECT(tSeg, TTPathSegment);

	      lastSeg->next = tSeg;
	      tSeg->next = NULL;
	      tSeg->start = p->backptr;
	      tSeg->length = curSeg->length - scnt - 1; // changed 2 -> 1
	      
	      /* Add to stack */
	      TTAddSegStack(&(tt_stack[ii+1]), 
			    (tt_stack_size-ii-1), newSeg, metric);
	      
#if (DEBUG > 10)
	      for (jj = 0; jj < tt_stack_size; ++jj)
		{
		  printf("TTStack: candidate %d metric = %f, path ptr 0x%X\n",
			 jj, tt_stack[jj].metric, tt_stack[jj].path);
		}
#endif
	      
	    }
	  
	} /* End loop over all the predecessor PLRs */

      //	  if (plr == curSeg->end)
      //	    break;
    }

  tt_last_path_found = candidate_num;
  return 1;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


void RCPCDecoder::TTAddSegStack(TTStack *pathStack, int stackSize, 
				TTPathSegment *seg,
				float metric)
{
  int loc;
  int N;
  int lower;
  int upper;


  /* quick check for top of stack */
  if (metric < pathStack[0].metric)
    {
      /* put it on the top of the stack */
      loc = 0;
    }
  else 
    {
      upper = stackSize - 1; lower = 0; loc = (stackSize) >> 1;
      while ((upper-lower) > 1)
	{
	  if (metric < pathStack[loc].metric)
	    { /* set the upper marker */
	      upper = loc;
	    }
	  else
	    { /* set the lower marker */
	      lower = loc;
	    }
	  loc = (upper + lower) >> 1;
	  
	}

      loc  = upper;
    }
  
  if (pathStack[stackSize - 1].path != NULL)
    { /* Free up last path segment if non NULL */
      FreeTTPathSegment(pathStack[stackSize - 1].path);
    }
  
  /* Now shift locations in the stack array */
  N = stackSize - 1 - loc;
  if (N)
    {
      memmove(&(pathStack[loc+1]), &(pathStack[loc]), N*sizeof(TTStack));
    }
  
  pathStack[loc].path = seg;
  pathStack[loc].metric = metric;	  

}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

void RCPCDecoder::FreeTTPathSegment(TTPathSegment *seg)
{
  TTPathSegment *next;

  while (seg != NULL)
    {
      next = seg->next;
      DELETE_OBJECT(seg);
      seg = next;
    }
}


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

void RCPCDecoder::decode_pending(BitBuffer *out_buffer)
{
  /* flush out the decision buffer.
   * Will find the state with the lowest metric and 
   * decode backwards from that state.
   */
  int ii;
  SurvivorType *ptr, *current_stage;
  float min_metric;
  int min_state;
  int buffer_size;

  if (num_stages)
    {
      current_stage = path_buffer[0][pathseg_idx];
      buffer_size = num_stages;
    }
  else
    {
      current_stage = path_buffer[pathbuf_idx][pathseg_idx];
      buffer_size = (pathbuf_idx*pathseg_size) + pathseg_idx +1;
    }

  min_metric = current_stage[0].metric;

    
  min_state = 0;

  if (tt_mode)
    {
      for (ii = 2; ii < (num_states<<1); ii+=2)
	{
	  if (current_stage[ii].metric < min_metric)
	    {
	      min_state = ii;
	      min_metric = current_stage[ii].metric;
	    }
	}
    }
  else
    {
      for (ii = 1; ii < num_states; ++ii)
	{
	  if (current_stage[ii].metric < min_metric)
	    {
	      min_state = ii;
	      min_metric = current_stage[ii].metric;
	    }
	}
    }

#if DEBUG
  printf("Min metric is %f\n", min_metric);
#endif

  ptr = current_stage + min_state;

  recurse_buffer(ptr, buffer_size - 2, out_buffer);

}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int RCPCDecoder::decode_bits(int word, int bits, BitBuffer *out_buffer, int flag)
{
  float metric;
  float bmetric;
  float norm_value;
  float min_metric;
  int min_path;
  int ii, jj, max_jj;
  int mask;
  int label;
  int bcnt;

  int state;
  int bit;

  SurvivorType *ptr;
  SurvivorType *current_stage, *next_stage;

  bcnt = bits;


  if (flag == TRELLIS_INIT)
    {
      memset(token_buffer, 0, num_states*sizeof(int));
      token_buffer[0] = 1;
    }
	

  while (bits)
    {
      /* put the bits into current until it has enough */
      for (mask = (1 << (bits-1)); bits && bits_for_transition;
	   --bits_for_transition, --bits, mask >>=1)
	{
	  current <<= 1;
	  if (mask & word) current |= 1;
	  ++loc;
	}

      if (!bits_for_transition)
	{
	  /* init temp buffer metrics */
	  for (ii = 0; ii < stage_size; ++ii)
	    working_trellis_stage[ii] = SurvivorType(-1, INIT_METRIC, NULL);
	  min_metric  = INIT_METRIC;
	  min_path = -1;

	  memset(new_token, 0, num_states*sizeof(int));

	  /* loop over all state transitions 
	   * and compute the branch metrics
	   */
	  if (num_stages)
	    {
	      current_stage = (path_buffer[0][pathseg_idx]);
	      pathseg_idx = (pathseg_idx + 1) % num_stages;
	      next_stage = (path_buffer[0][pathseg_idx]);

	    }
	  else
	    { // allocate next stage
	      current_stage = path_buffer[pathbuf_idx][pathseg_idx];
	      if ((++pathseg_idx) >= pathseg_size)
		{
		  if ((++pathbuf_idx) >= pathbuf_size)
		    {
		      ErrorMsgAndExit("RCPCDecoder", "decode_bits",
			       "Out of buffer space in RCPC decoder.");
		    }
		  // allocate new pathseg array
		  NEW_ARRAY(path_buffer[pathbuf_idx], pathseg_size, 
					SurvivorType *);

		  pathseg_idx = 0;
			     
		}
	      
	      NEW_ARRAY(next_stage, stage_size, SurvivorType);

	      path_buffer[pathbuf_idx][pathseg_idx] = next_stage;


	    }

	  max_jj = (flag == TRELLIS_FLUSH) ? 1 : 2;
	  
	  if (!tt_mode)
	    {
	      for (ii = 0; ii < num_states; ++ii)
		{
		  if (token_buffer[ii])
		    {
		      metric = current_stage[ii].metric;
#if (DEBUG > 10)
		      printf("metric at %d  = %f\n", ii, metric);
#endif
		      // compute the normalization value for this state
		      norm_value = compute_normalization(ii, current);

		      for (jj = 0; jj < max_jj; ++jj)
			{
			  /* get the relevant label */
			  label = labels[(ii<<1) + jj];

			  state = ((ii << 1) | jj) & state_mask;			  
			  bmetric = compute_metric(current, punc_entry, label, 
						   ii, state);
			  bmetric -= norm_value;
			  
			  /* add to current state metric and forward to next
			   * state 
			   */
			  bmetric += metric;

			  
			  if (working_trellis_stage[state].metric > bmetric)
			    { /* update the survivor info for this entry */
			      working_trellis_stage[state].metric = bmetric;
			      working_trellis_stage[state].backptr = 
				current_stage + ii;
			      working_trellis_stage[state].state = state;
			      new_token[state] =1;
			    }
			  
			  if (bmetric < min_metric)
			    {
			      min_metric = bmetric;
			      min_path = state;
			    }
			}
		    }
		}

	    }

	  else
	    { // Tree-Trellis mode
	      for (ii = 0; ii < num_states; ++ii)
		{
		  if (token_buffer[ii])
		    {
		      metric = current_stage[ii << 1].metric;
#if (DEBUG > 10)
		      printf("metric at %d  = %f\n", ii, metric);
#endif
		      // compute the normalization value for this state
		      norm_value = compute_normalization(ii, current);

		      for (jj = 0; jj < max_jj; ++jj)
			{
			  /* get the relevant label */
			  label = labels[(ii<<1) + jj];

			  state = ((ii << 1) | jj) & state_mask;			  
			  bmetric = compute_metric(current, punc_entry, label, 
						   ii, state);

			  bmetric -= norm_value;
			  
			  /* add to current state metric and forward to next
			   * state 
			   */
			  bmetric += metric;
			  
			  if (working_trellis_stage[(state<<1)].metric > bmetric)
			    { /* update the survivor info for this entry */
			      working_trellis_stage[(state<<1)+1] =  
				working_trellis_stage[(state<<1)]; 
			      working_trellis_stage[state<<1].metric = bmetric;
			      working_trellis_stage[state<<1].backptr = 
				current_stage + (ii<<1);
			      working_trellis_stage[state<<1].state = state;
			      new_token[state] =1;
			    }
			  else
			    { // place info in backup
			      working_trellis_stage[(state<<1)+1].metric = 
				bmetric;
			      working_trellis_stage[(state<<1)+1].backptr = 
				current_stage + (ii << 1);
			      working_trellis_stage[(state<<1)+1].state = state;
			    }

			  
			  if (bmetric < min_metric)
			    {
			      min_metric = bmetric;
			      min_path = (state << 1);
			    }
			}
		    }
		}


	    }

#if (DEBUG > 10)
	  printf("min metric = %f, min_path = %d\n", 
		 min_metric, min_path);
	  
	  for (ii = 0; ii < num_states; ++ii)
	    printf("%d ", token_buffer[ii]);
	  printf("\n");

	  for (ii = 0; ii < num_states; ++ii)
	    printf("%d ", new_token[ii]);
	  printf("\n");

#endif


	  memcpy(token_buffer, new_token, num_states*sizeof(int));
	  memcpy(next_stage, working_trellis_stage, 
		 stage_size*sizeof(SurvivorType));

	  if (num_stages)
	    { // possibly decode a bit if the buffer is full
	      if (!buffer_full && next_stage == path_buffer[0][0])
		{
		  buffer_full = 1;
#if DEBUG
		  printf("BUFFER FULL\n");
#endif
		}

	      if (buffer_full)
		{ /* decode beginning of buffer */
		  /* trace back the pointer to start of path */
		  for (ptr = working_trellis_stage + min_path, ii = 0;
		       ii < num_stages - 1; ++ii)
		    ptr = ptr->backptr;
		  
		  /* figure out the input bit */
		  bit = ptr->state & 1;
		  
		  out_buffer->buffer_bit(bit);
#if (DEBUG > 10)
		  printf("Sent bit %d to output\n", bit);
#endif
		  
		}
	    }

	  // check if location exceeds sched spec
	  if (rate_sched && sched_idx >= 0 && 
	      (loc >= rate_sched->sched_array[sched_idx].location))
	    {
	      code_num = rate_sched->sched_array[sched_idx].code_num;
	      punc_ptr = code_spec.PunctMat + (code_num*code_spec.Period);
	      // Set sched_idx to -1 if at the last code so we no longer
	      // worry about code changes
#if DEBUG
	      printf("RCPCDecoder: changed code num to %d at loc %d\n",
		     code_num, loc);
#endif
	      sched_idx = (sched_idx < rate_sched->size - 1) ? sched_idx+1 : -1;
	    }

	  

	  /* increment per_idx and get next bits for transition */
	  per_idx = (per_idx + 1) % code_spec.Period;
	  /* build the relevant puncture entry */
	  punc_entry = punc_ptr[per_idx];
	  bits_for_transition = weight(punc_entry);
	  current = 0;
	  
	}

    }
  

  return (bcnt-bits);

}


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int RCPCDecoder::decode_bits(int bits, BitBuffer *input, 
			     BitBuffer *out_buffer, int flag)
{
  float metric;
  float bmetric;
  float norm_value;
  float min_metric;
  int min_path;
  int ii, jj, max_jj;
  int word;
  int label;
  int bcnt;

  int state;
  int bit;

  SurvivorType *ptr;
  SurvivorType *current_stage, *next_stage;

  bits = Max(Min(bits, input->rbits_to_end()), 0);
  bcnt = bits;

  if (flag == TRELLIS_INIT)
    {
      memset(token_buffer, 0, num_states*sizeof(int));
      token_buffer[0] = 1;
    }
	

  while (bits)
    {
#if (DEBUG > 10)
      printf("Bits for transition = %d\n", bits_for_transition);
#endif
      /* put the bits into current until it has enough */
      for ( ; bits && bits_for_transition;
	   --bits_for_transition, --bits)
	{
	  if ((word = input->read_bits(1)) < 0)
	    return bcnt - bits;
	  current <<= 1;
	  if (word) current |= 1;
	  ++loc;
	}

#if (DEBUG > 10)
      printf("After getting bits for transition loc = %d\n", loc);
#endif


      if (!bits_for_transition)
	{


	  /* init temp buffer metrics */
	  for (ii = 0; ii < stage_size; ++ii)
	    working_trellis_stage[ii] = SurvivorType(-1, INIT_METRIC, NULL);
	  min_metric  = INIT_METRIC;
	  min_path = -1;

	  memset(new_token, 0, num_states*sizeof(int));

	  /* loop over all state transitions 
	   * and compute the branch metrics
	   */
	  if (num_stages)
	    {
	      current_stage = (path_buffer[0][pathseg_idx]);
	      pathseg_idx = (pathseg_idx + 1) % num_stages;
	      next_stage = (path_buffer[0][pathseg_idx]);

	    }
	  else
	    { // allocate next stage
	      current_stage = path_buffer[pathbuf_idx][pathseg_idx];
	      if ((++pathseg_idx) >= pathseg_size)
		{
		  if ((++pathbuf_idx) >= pathbuf_size)
		    {
		      ErrorMsgAndExit("RCPCDecoder", "decode_bits",
				      "Out of buffer space in RCPC decoder.");
		    }
		  // allocate new pathseg array
		  NEW_ARRAY(path_buffer[pathbuf_idx], pathseg_size, 
					SurvivorType *);

		  pathseg_idx = 0;
			     
		}
	      
	      NEW_ARRAY(next_stage, stage_size, SurvivorType);

	      path_buffer[pathbuf_idx][pathseg_idx] = next_stage;
	      

	    }

	  max_jj = (flag == TRELLIS_FLUSH) ? 1 : 2;

	  if (!tt_mode)
	    {
	      for (ii = 0; ii < num_states; ++ii)
		{
		  if (token_buffer[ii])
		    {
		      metric = current_stage[ii].metric;
#if (DEBUG > 10)
		      printf("metric at %d  = %f\n", ii, metric);
#endif
		      // compute the normalization value for this state
		      norm_value = compute_normalization(ii, current);

		      for (jj = 0; jj < max_jj; ++jj)
			{
			  /* get the relevant label */
			  label = labels[2*ii + jj];

			  state = ((ii << 1) | jj) & state_mask;
			  
			  bmetric = compute_metric(current, punc_entry, label,
						   ii, state);

			  bmetric -= norm_value;
			  
			  /* add to current state metric and forward to next
			   * state 
			   */
			  bmetric += metric;

			  
			  if (working_trellis_stage[state].metric > bmetric)
			    { /* update the survivor info for this entry */
			      working_trellis_stage[state].metric = bmetric;
			      working_trellis_stage[state].backptr = 
				current_stage + ii;
			      working_trellis_stage[state].state = state;
			      new_token[state] =1;
			    }
			  
			  if (bmetric < min_metric)
			    {
			      min_metric = bmetric;
			      min_path = state;
			    }
			}
		    }
		}

	    }

	  else
	    { // Tree-Trellis mode
	      for (ii = 0; ii < num_states; ++ii)
		{
		  if (token_buffer[ii])
		    {
		      metric = current_stage[ii << 1].metric;
#if (DEBUG > 10)
		      printf("TT mode metric at %d  = %f\n", ii, metric);
#endif
		      // compute the normalization value for this state
		      norm_value = compute_normalization(ii, current);

		      for (jj = 0; jj < max_jj; ++jj)
			{
			  /* get the relevant label */
			  label = labels[(ii<<1) + jj];

			  state = ((ii << 1) | jj) & state_mask;			  
			  bmetric = compute_metric(current, punc_entry, label, 
						   ii, state);
			  bmetric -= norm_value;
			  /* add to current state metric and forward to next
			   * state 
			   */
			  bmetric += metric;

			  
			  if (working_trellis_stage[(state<<1)].metric > bmetric)
			    { /* update the survivor info for this entry */
			      working_trellis_stage[(state<<1)+1] =  
				working_trellis_stage[(state<<1)]; 
			      working_trellis_stage[state<<1].metric = bmetric;
			      working_trellis_stage[state<<1].backptr = 
				current_stage + (ii<<1);
			      working_trellis_stage[state<<1].state = state;
			      new_token[state] =1;
			    }
			  else
			    { // place info in backup
			      working_trellis_stage[(state<<1)+1].metric = 
				bmetric;
			      working_trellis_stage[(state<<1)+1].backptr = 
				current_stage + (ii << 1);
			      working_trellis_stage[(state<<1)+1].state = state;
			    }

			  
			  if (bmetric < min_metric)
			    {
			      min_metric = bmetric;
			      min_path = (state<<1);
			    }
			}
		    }
		}


	    }



#if (DEBUG > 10)
	  if (min_metric > 0)
	    printf("min metric = %f, min_path = %d, bit number %d\n", 
		   min_metric, min_path, bcnt - bits);
#endif

	  memcpy(token_buffer, new_token, num_states*sizeof(int));
	  memcpy(next_stage, working_trellis_stage, 
		 stage_size*sizeof(SurvivorType));

	  if (num_stages)
	    { // possibly decode a bit if the buffer is full
	      if (!buffer_full && next_stage == path_buffer[0][0])
		{
		  buffer_full = 1;
#if DEBUG
		  printf("BUFFER FULL\n");
#endif
		}

	      if (buffer_full)
		{ /* decode beginning of buffer */
		  /* trace back the pointer to start of path */
		  for (ptr = working_trellis_stage + min_path, ii = 0;
		       ii < num_stages - 1; ++ii)
		    ptr = ptr->backptr;
		  
		  /* figure out the input bit */
		  bit = ptr->state & 1;
		  
		  out_buffer->buffer_bit(bit);
#if (DEBUG > 10)
		  printf("Sent bit %d to output\n", bit);
#endif
		  
		}
	    }


	  // check if location exceeds sched spec
	  if (rate_sched && sched_idx >= 0 && 
	      (loc >= rate_sched->sched_array[sched_idx].location))
	    {
	      code_num = rate_sched->sched_array[sched_idx].code_num;
	      punc_ptr = code_spec.PunctMat + (code_num*code_spec.Period);
	      // Set sched_idx to -1 if at the last code so we no longer
	      // worry about code changes
#if DEBUG
	      printf("RCPCDecoder: changed code num to %d at loc %d\n",
		     code_num, loc);
#endif
	      sched_idx = (sched_idx < rate_sched->size - 1) ? sched_idx+1 : -1;
	    }

	  /* increment per_idx and get next bits for transition */
	  per_idx = (per_idx + 1) % code_spec.Period;
	  /* build the relevant puncture entry */
	  punc_entry = punc_ptr[per_idx];
	  bits_for_transition = weight(punc_entry);
	  current = 0;
	  
	}

    }
  

  return bcnt - bits;

}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int RCPCDecoder::max_code_index(void)
{
  return (code_spec.NumCodes-1);
}



// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

void RCPCDecoder::select_code(int code_number)
{


  if (code_number < 0 || code_number >= code_spec.NumCodes)
    {
      ErrorMsg("RCPCDecoder","select_code", "code_number %d out of range [0, %d]", code_number, code_spec.NumCodes-1);
      return;
    }
  code_num = code_number;
  punc_ptr = code_spec.PunctMat + (code_num*code_spec.Period);
  if (bits_for_transition == weight(punc_entry))
    { // at transition boundary so change
      punc_entry = punc_ptr[per_idx];
      bits_for_transition = weight(punc_entry);
    }

}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

float RCPCDecoder::compute_metric(int recvd, unsigned int punc, int label,
				  int start_state, int end_state)
{
  /* this routine computes the hamming distance
   * between the received word and the branch label 
   */
  
  int ii, tmp;
  float dist = 0;
  int plabel = 0;

  // to keep the compiler from issuing warnings
  start_state = start_state;
  end_state = end_state;

  /* form the punctured label */
  for (ii = 0; punc; punc >>= 1, label >>= 1)
    {
      if (punc & 1)
	{
	  plabel |= (label & 1) << ii;
	  ++ii;
	}
    }

  tmp = plabel ^ recvd;
  
  /* count number of ones in tmp */
  dist = weight(tmp);

  return dist;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

void RCPCDecoder::decode_TTpath(TTPathSegment *TTptr, 
				BitBuffer *out_buffer, int num_bits)
{
  int bit_pos;
  int ii;
  int m;
  int nwords;
  int wcnt;
  int wsize;
  int seg_cnt;
  long *buf;
  SurvivorType *sptr;
  TTPathSegment *tt_ptr;

  if (num_bits <= 0)
    { // determine the number of bits to decode
      num_bits = 0;
      for (tt_ptr = TTptr; tt_ptr != NULL; num_bits += tt_ptr->length, 
	     tt_ptr = tt_ptr->next)
	{
#if (DEBUG)
	  printf("decode_TTpath: segment length = %d, num_bits %d\n",
		 tt_ptr->length, num_bits);
	  /* check integrity */
#endif
	}
    }
      
#if (DEBUG)
  printf("decode_TTpath: num_bits = %d, path ptr = 0x%X\n", num_bits, TTptr);
#endif

  wsize = sizeof(long) << 3; // number of bits in 'long' data type

  // compute the amount of storage needed 
  nwords = (num_bits/wsize) + ((num_bits % wsize) != 0);
  
  // Allocate a temp buffer 
  NEW_ARRAY(buf, nwords, long);
  // set the buffer to zero
  memset(buf, 0, nwords*sizeof(long));

  bit_pos = (wsize - (num_bits % wsize)) % wsize;
  m = 1;
  tt_ptr = TTptr;
  sptr = tt_ptr->start;
  seg_cnt = tt_ptr->length;
  for (ii = 0, wcnt = nwords-1; ii < num_bits && sptr; ++ii, m <<= 1,
	 bit_pos++)
    {
      if (bit_pos >= wsize)
	{
	  m = 1; bit_pos = 0;wcnt--;
	}

      if (sptr->state & 1) buf[wcnt] |= m;

      if (!(--seg_cnt))
	{ // get next segment
	  tt_ptr = tt_ptr->next;
#if DEBUG
	  printf("Changing segments Current state %d\n", sptr->state);
	  if (sptr->backptr)
	    printf("Changing segments other metric is %f at state %d\n",
		   sptr->backptr->metric, sptr->backptr->state);
#endif
	  sptr = (SurvivorType *) ((tt_ptr != NULL) ? tt_ptr->start : NULL);
	  seg_cnt = (tt_ptr != NULL) ? tt_ptr->length : 0;
	  if (tt_ptr != NULL)
	    {

#if (DEBUG )
	      printf("TTdecode: Got next segment of len %d metric %f and state %d ptr 0x%X\n",
		     seg_cnt, sptr->metric, sptr->state, sptr);
#endif
	    }
	}
      else
	{
	  sptr = sptr->backptr;
	}
    }

  m = ii; // m holds the number of bits now

#if (DEBUG)
  printf("Within TT_decode: bit_pos = %d, wcnt = %d\n", bit_pos, wcnt);
#endif

  if (bit_pos != wsize)
    {
      buf[wcnt] <<= wsize - bit_pos;
      out_buffer->buffer_bits(bit_pos, buf[wcnt]);
      m -= bit_pos;
    }
  else 
    {
      out_buffer->buffer_bits(wsize, buf[wcnt]);
      m -= wsize;
    }

  // now output to buffer 
  for (ii = wcnt+1; ii < (nwords - 1); ++ii)
    {
      out_buffer->buffer_bits(wsize, buf[ii]);
      m -= wsize;
    }
  
  // output final word
  out_buffer->buffer_bits(m, buf[nwords-1]);

#if (DEBUG )
    printf("Within TT_decode: bits in buffer = %d\n", out_buffer->bits_in_buffer());
#endif

  DELETE_ARRAY(buf);
}


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
int RCPCDecoder::compute_coded_bits(int raw_bits, int code_num)
{
  int ii;
  int bits_per_period;
  int num_periods;
  int total_bits;
  int *punc_ptr;

  if (code_num < 0 || code_num >= code_spec.NumCodes)
    {
      ErrorMsg("RCPCDecoder", "compute_coded_bits", "code_number %d out of range [%d, %d]\n", code_num, 0, code_spec.NumCodes-1);
      return -1;
    }


  num_periods = raw_bits/code_spec.Period;

  punc_ptr = code_spec.PunctMat + (code_num*code_spec.Period);
  for (bits_per_period = 0, ii = 0; ii < code_spec.Period; ++ii)
    {
      bits_per_period += weight(punc_ptr[ii]);
    }

  total_bits = bits_per_period*num_periods;


  // get the remaining bits
  for (ii = 0; ii < (raw_bits % code_spec.Period); ++ii)
    {
      total_bits += weight(punc_ptr[ii]);
    }

  return total_bits;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
int RCPCDecoder::compute_coded_bits(int raw_bits)
{
  return compute_coded_bits(raw_bits, code_num);

}



// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


