// -*- c++ -*-

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

//             F E C   R O U T I N E S

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
//
// P. Greg Sherwood            sherwood@code.ucsd.edu
// Kenneth Zeger	       zeger@code.ucsd.edu
//
// 9500 Gilman Dr MC 0407
// La Jolla, CA 92093
//
//
// Copyright (c) 1997 P. Greg Sherwood and Kenneth Zeger
//
// This program is Copyright (c) by P. Greg Sherwood and Kenneth Zeger
// It may not be redistributed without written permission of its 
// copyright holders. It may not be sold for profit or
// incorporated in commercial programs without the written permission
// of the copyright holders. This program is provided as is, without any
// express or implied warranty, without even the warranty of fitness
// for a particular purpose.
//

// - - Inclusion - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#include "fec.h"


// - - External definitions  - - - - - - - - - - - - - - - - - - - - - -


#define LOWER_MASK 0x0F

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

/* CRC info can be found in the paper in August 1988 IEEE Micro pp.~64-71+ */
    
static int CRC_16_table[] = {0xA001, 0xF001, 0xD801, 0xCC01, 0xC601, 
			     0xC301, 0xC181, 0xC0C1};

// Note: CRC-16 polynomial is X^16 + X^15 + X^2 + 1
int compute_crc_16(BitBuffer *input, int num_bits)
{

  int crc_reg = 0;
  int m, t, word, bits;

  while (num_bits)
    {
      bits = Min(num_bits, 8);

      if (input->read_bits(bits, word) != bits)
	{
	  ErrorMsg("FEC", "compute_crc_16", 
		   "Error reading bits");
	}

      num_bits -= bits;
      
      t = word ^ crc_reg;

      crc_reg >>= 8;

      for (m = 7; m >= 0 ; --m, t >>= 1)
	{
	  if (t & 1)
	    crc_reg ^= CRC_16_table[m];
	}
    }
  // return lower 16 bits 
  return (crc_reg);
}

static int CRC_12_table[] = {0xF01, 0x881, 0xB41, 0xAA1, 0xA51, 
			     0xA29, 0xA15, 0xA0B};

// Note: CRC-12 polynomial is X^12 + X^11 + X^3 + X^2 + X + 1
int compute_crc_12(BitBuffer *input, int num_bits)
{

  int crc_reg = 0;
  int m, t, word, bits;

  while (num_bits)
    {
      bits = Min(num_bits, 8);

      if (input->read_bits(bits, word) != bits)
	{
	  ErrorMsg("FEC", "compute_crc_12", 
		   "Error reading bits");
	}

      num_bits -= bits;
      
      t = word ^ crc_reg;

      crc_reg >>= 8;

      for (m = 7; m >= 0 ; --m, t >>= 1)
	{
	  if (t & 1)
	    crc_reg ^= CRC_12_table[m];
	}
    }
  // return lower 16 bits 
  return (crc_reg);
}

// NOTE : LRCC-8 polynomial is X^8+1
int compute_crc_8(BitBuffer *input, int num_bits)
{

  int crc_reg = 0;
  int word, bits;

  while (num_bits)
    {
      bits = Min(num_bits, 8);

      if (input->read_bits(bits, word) != bits)
	{
	  ErrorMsg("FEC", "compute_crc_8", 
		   "Error reading bits");
	}

      num_bits -= bits;

      crc_reg ^= word;
      
    }
  // return lower 8 bits 
  return (crc_reg);
}

// This function computes the reduced lookup table for byte-processing
// of a crc of degree up to 32.
int compute_reduced_table(unsigned long crc_generator, int *reduced_table)
{
  unsigned long rem, tmp, power;

  rem = crc_generator;
  
  if (!reduced_table)
    return 0;
  
  for (power = 0; power < 8; ++power)
    {
      reduced_table[power] = rem;
      tmp = rem & 0x1;
      rem >>= 1;
      if (tmp)
	rem ^= crc_generator;
    }

  return 1;

}

int compute_crc_16(BitBuffer *input, int num_bits, int *reduced_table)
{

  int crc_reg = 0;
  int m, t, word, bits;

  while (num_bits)
    {
      bits = Min(num_bits, 8);

      if (input->read_bits(bits, word) != bits)
	{
	  ErrorMsg("FEC", "compute_crc_16", 
		   "Error reading bits: buffer contains %d bits\n",
		   input->rbits_to_end());
	}

      num_bits -= bits;
      
      t = word ^ crc_reg;

      crc_reg >>= 8;

      // added check to see if t is non-zero - if not then exit loop
      for (m = 7; (m >= 0) && t ; --m, t >>= 1)
	{
	  if (t & 1)
	    crc_reg ^= reduced_table[m];
	}
    }
  // return lower 16 bits 
  return (crc_reg);
}
