// -*- c++ -*-
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

//              B I T  B U F F E R   C L A S S

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
//
// P. Greg Sherwood            sherwood@code.ucsd.edu
// Kenneth Zeger	       zeger@code.ucsd.edu
//
// 9500 Gilman Dr MC 0407
// La Jolla, CA 92093
//
//
// Copyright (c) 1997 P. Greg Sherwood and Kenneth Zeger
//
// This program is Copyright (c) by P. Greg Sherwood and Kenneth Zeger
// It may not be redistributed without written permission of its 
// copyright holders. It may not be sold for profit or
// incorporated in commercial programs without the written permission
// of the copyright holders. This program is provided as is, without any
// express or implied warranty, without even the warranty of fitness
// for a particular purpose.
//

#ifndef BitBuffer_H
#define BitBuffer_H

// - - Inclusion - - - - - - - - - - - - - - - - - - - - - - - - - - - -

//#include <stream.h>
#include <iostream>
#include <string.h>
#include "general.h"

#ifdef __IRIX__
#include "bool.h"
#endif

using namespace std;

// - - External function - - - - - - - - - - - - - - - - - - - - - - - -


// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

//  Class definitions

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

/* 
 *        NOTES:
 *
 *  When buffering multiple bits in a call to 'buffer_bits', the MSB
 *  is assumed to be the oldest bit (i.e. next bit in the bit stream).
 *  The internal storage of the bitbuffer is the same.  Also when data
 *  is written to a file this convention is used.  In diagram form we
 *  have:
 *
 *   MSB               LSB
 *    |_________________|
 *        time ->
 *
 *  Note: that in the original Said/Pearlman this convention is used 
 *  EXCEPT when writing bytes to a file.  There is a 'reverse' flag
 *  on the file I/O routines to allow reading/writing files for interfacing
 *  with the Said/Pearlman code. Also they don't reverse the first byte which
 *  is the ID word, so use the bit reversed ID if you're writing the file 
 *  in their format.
 *
 */


#define BB_NODE_SIZE 100
#define BITS_PER_WORD (sizeof(long) << 3)

#define BB_START 0
#define BB_END -1

#define BB_DUMP_SUCCESS 0
#define BB_DUMP_ESPEC -1
#define BB_DUMP_EFILE -2

#define ALL_BITS -1

typedef struct _BufferNode {
  unsigned long data[BB_NODE_SIZE];
  _BufferNode *next;
} BufferNode;

struct BufferPos {
  BufferNode *nodeptr;
  int word_idx;
  int bit_idx;
};


class BitBuffer
{
  // . constructor  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .

  public :

  BitBuffer(int preserve_on_read = 0);

  BitBuffer(const BitBuffer &srcbuf) ;

  // . destructor   .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .

  ~BitBuffer();

  // . class functions .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .

  long bytes_in_buffer(void) { return (bit_counter >> 3); }

  long bits_in_buffer(void) { return (bit_counter);}

  long bytes_to_readptr(void) {return ((bit_counter - rbits_from_end) >> 3); }

  long bits_to_readptr(void) {return (bit_counter - rbits_from_end);}

  long rbits_to_end(void) {return (rbits_from_end);}

  void buffer_bits(int numb_bits, int symb);

  void buffer_bit(int symb) {buffer_bits(1, symb);}

  bool set_bit(int bit_index, int bit_value);

  int read_bits(int numb_bits);

  int read_bits(int numb_bits, int& symb, int get_available = false);

  void empty_buffer();

  int set_preserve(int preserve_on_read); // returns the old state

  int position_read_ptr(int bit_pos);

  bool concat(BitBuffer& src_buf);

  friend BitBuffer extract_range(BitBuffer& src_buf, 
				 int start_pos = BB_START, 
				 int end_pos = BB_END);

  int fill_from_file(FILE *fp, int number_of_bytes = ALL_BITS, 
		     bool reverse = false);

  int dump_bits(FILE *fp, int start_pos = BB_START, int end_pos = BB_END,
	    bool reverse = false);

  int print_bits(int start_pos = BB_START, int end_pos = BB_END,
		 ostream& stream = cout);

  int hamming_weight(int start_pos = BB_START, int end_pos = BB_END);

  bool copy_range(BitBuffer& src_buf, 
		  int start_pos = BB_START, 
		  int end_pos = BB_END);

  int compare(BitBuffer& bufB);

  // . operators    .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .

  int operator[](int index);

  friend ostream&    operator << (ostream& s, BitBuffer&);

  BitBuffer & operator =(const BitBuffer &src);

  BitBuffer& operator += (unsigned int newBit);

  BitBuffer& operator += (BitBuffer& src_buf);

  int h_xor(BitBuffer& xor_buf);


  // conditional test operators
  // <= tests if bufA is a prefix of bufB
  // >= tests if bufB is a prefix of bufA
  friend bool operator <= (BitBuffer& bufA, BitBuffer& bufB);
  friend bool operator >= (BitBuffer& bufA, BitBuffer& bufB);
  friend bool operator == (BitBuffer& bufA, BitBuffer& bufB);
  friend bool operator != (BitBuffer& bufA, BitBuffer& bufB);


  // . class data   .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .

  private :

  // The BufferNode pointers contain the two ends of the 
  // linked list holding the data and the current node 
  // for reading.  Bits are added at buffer_end and read from
  // read_ptr
  BufferNode *buffer_start, *buffer_end, *read_ptr;



  // These indices contain the state information for the bit 
  // location of the read/write pointer.

  // in_word_index =	index within BufferNode array indicating the 
  //			the location of the current word for writing.
  // in_bit_index =	index indicating the number of bits  
  //			stored in the current packed word used for writing.
  // out_word_index =	index within BufferNode array indicating the 
  //			the offset in the buffer_start BufferNode where
  //			data begins.  This is the read position when 
  //			the 'preserve' flag is false.
  // out_bit_index =	index indicating the offset into the buffer_start
  //			word (in bits). This is the read position when 
  //			the 'preserve' flag is false.
  // read_word_index =	indicates word offset in the BufferNode array for
  //			reading when the 'preserve' flag is true.
  // read_bit_index =	indicates bit offset when for reading when the 
  //			'preserve' flag is true.
  //
  int in_word_index; 
  int in_bit_index;
  int out_word_index;
  int out_bit_index;
  int read_word_index;
  int read_bit_index;
  
  // A flag to indicate whether to keep bits after they have been read.
  int preserve;

  // counter for number in bits in the buffer
  long bit_counter;

  // the distance of the read pointer from the end of the buffer (in bits).
  long rbits_from_end;

  // private class routines.
  struct BufferPos find_pos(int index);

  void copy_values(const BitBuffer &buf);

}; // end of class < BitBuffer >


inline BitBuffer& BitBuffer::operator =(const BitBuffer &src)
{
  this->copy_values(src);

  return *this;
}


inline BitBuffer& BitBuffer::operator += (unsigned int newBit)
{
  this->buffer_bits(1, newBit);

  return *this;
}

inline BitBuffer& BitBuffer::operator += (BitBuffer& src_buf)
{
  this->concat(src_buf);
  return *this;
}

inline bool operator >= (BitBuffer& bufA, BitBuffer& bufB)
{
  return (bufB <= bufA);
}

inline bool operator == (BitBuffer& bufA, BitBuffer& bufB)
{
  return ((bufA <= bufB) && 
	  (bufA.bits_in_buffer() == bufB.bits_in_buffer()));
}


inline bool operator != (BitBuffer& bufA, BitBuffer& bufB)
{
  return (!(bufA == bufB));
}




// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

//  Related functions

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =




int move_bits(BitBuffer *dest, BitBuffer *src, int num_bits = ALL_BITS);

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

#endif
// end of file < BinCode.H >










