// -*- c++ -*-
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

//              B I T  B U F F E R  C L A S S

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
//
// P. Greg Sherwood            sherwood@code.ucsd.edu
// Kenneth Zeger	       zeger@code.ucsd.edu
//
// 9500 Gilman Dr MC 0407
// La Jolla, CA 92093
//
//
// Copyright (c) 1997 P. Greg Sherwood and Kenneth Zeger
//
// This program is Copyright (c) by P. Greg Sherwood and Kenneth Zeger
// It may not be redistributed without written permission of its 
// copyright holders. It may not be sold for profit or
// incorporated in commercial programs without the written permission
// of the copyright holders. This program is provided as is, without any
// express or implied warranty, without even the warranty of fitness
// for a particular purpose.
//

// - - Inclusion - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#include "bitbuffer.h"


// - - Internal definitions  - - - - - - - - - - - - - - - - - - - - - -

static const unsigned long scInitBmask = (1 << (BITS_PER_WORD - 1));
static void reverse_byte(int& word);

#define IN_BIT_MASK 0x1

// - - External function - - - - - - - - - - - - - - - - - - - - - - - -


// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

//  Member functions of the class < BitBuffer >

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

BitBuffer::BitBuffer(int preserve_on_read) 
{
  NEW_OBJECT(buffer_start, BufferNode);

  buffer_end = read_ptr = buffer_start; 
  buffer_start->next = NULL;
  in_word_index = 0; out_word_index = 0; rbits_from_end = bit_counter = 0;
  read_word_index = 0;
#if DEBUG
  printf("Within constructor, bit counter = %d\n", bit_counter);
#endif
  in_bit_index =  0;

  read_bit_index = out_bit_index = 0;

  preserve = preserve_on_read;
}

//----------------------------------------------------------------------

// copy constructor
BitBuffer::BitBuffer(const BitBuffer &srcbuf) 
{
  NEW_OBJECT(buffer_start, BufferNode);

  buffer_end = read_ptr = buffer_start; 
  buffer_start->next = NULL;

  // copy values
  copy_values(srcbuf);

}


//----------------------------------------------------------------------

void BitBuffer::copy_values(const BitBuffer &buf)
{
  BufferNode *srcptr, *tmp;
  int bit_pos;

  // first empty the existing buffer
  empty_buffer();

  in_word_index = buf.in_word_index; 
  in_bit_index = buf.in_bit_index;
  out_word_index = buf.out_word_index; 
  out_bit_index = buf.out_bit_index;
  read_word_index = buf.read_word_index;
  read_bit_index = buf.read_bit_index;
  rbits_from_end = buf.rbits_from_end;
  bit_counter = buf.bit_counter;

  preserve = buf.preserve;


  for (srcptr = buf.buffer_start; 
       srcptr && (srcptr != buf.buffer_end); 
       srcptr = srcptr->next)
    {
      // copy the contents 
      memcpy(buffer_end->data, srcptr->data, sizeof(srcptr->data));
      
      if ((tmp = buffer_end->next) == NULL)
	{ // allocate more memory
	  NEW_OBJECT(tmp, BufferNode);
	  tmp->next = NULL;
	  buffer_end->next = tmp;
	}
      buffer_end = tmp;
    }

  // copy the last node
  memcpy(buffer_end->data, srcptr->data, sizeof(srcptr->data));

  if (preserve)
    { // set the location of the readptr
      bit_pos = bit_counter - rbits_from_end;
      position_read_ptr(bit_pos);
    }

	  
}

//----------------------------------------------------------------------

BitBuffer::~BitBuffer()
{
  BufferNode *tmp;
  while (buffer_start != buffer_end)
    {
      tmp = buffer_start->next;
      delete buffer_start;
      buffer_start = tmp;
    }
  delete buffer_start;

}

//----------------------------------------------------------------------

void BitBuffer::empty_buffer()
{
  BufferNode *tmp;
  while (buffer_start != buffer_end)
    {
      tmp = buffer_start->next;
      delete buffer_start;
      buffer_start = tmp;
    }

  read_ptr = buffer_start;
  read_word_index = 0;
  in_word_index = 0; out_word_index = 0; 
  rbits_from_end = bit_counter = 0;
  in_bit_index =  0;
  read_bit_index = out_bit_index = 0;

}

//----------------------------------------------------------------------

/*
 * Bits are packed into words starting from the MSB and
 * proceeding to the LSB.  This is the same convention 
 * used in reading multi-bit words passed into this procedure.
 *
 *   MSB               LSB
 *    |_________________|
 *        time ->
 */

void BitBuffer::buffer_bits(int numb_bits, int symb)
{
  unsigned long bmask = (scInitBmask >> in_bit_index);
  for (unsigned long m = 1 << (numb_bits - 1); m > 0; m >>= 1) {
    if (symb & m) buffer_end->data[in_word_index] |= bmask;
    else buffer_end->data[in_word_index] &= (~bmask);
    // update counter state
    bit_counter++; ++rbits_from_end; ++in_bit_index;
    if (in_bit_index >= BITS_PER_WORD) 
      {
	// if (buffer_start == buffer_end && in_word_index == out_word_index)
	// out_bit_index = BITS_PER_WORD;
	if ((++in_word_index) >= BB_NODE_SIZE)
	  { 
	    BufferNode *tmp;
	    if ((tmp = buffer_end->next) == NULL)
	      { // allocate a new node
		NEW_OBJECT(tmp, BufferNode);
		tmp->next = NULL;
		buffer_end->next = tmp;
	      }
	    buffer_end = tmp;
	    in_word_index = 0;
	  }
	in_bit_index = 0;
	bmask = scInitBmask;
      }
    else
      {
	bmask >>= 1; // update mask to the next input bit position
      }
  }
}

//----------------------------------------------------------------------

int BitBuffer::set_preserve(int preserve_on_read)
{
  int old_preserve = preserve;

  preserve = (preserve_on_read != 0);
  read_ptr = buffer_start;
  read_word_index = out_word_index;
  read_bit_index = out_bit_index;
  rbits_from_end = bit_counter;
  return old_preserve;
}

//----------------------------------------------------------------------

struct BufferPos BitBuffer::find_pos(int index)
{
  BufferNode *ptr;
  int ii, num;
  struct BufferPos pos = {NULL, 0, 0};

  if (index >= bit_counter)
    return pos;

  // find position from start
  index = index + out_word_index*BITS_PER_WORD + 
    (out_bit_index);

  // compute the number of nodes to skip
  num = index/(BITS_PER_WORD*BB_NODE_SIZE);

  for (ptr = buffer_start, ii = 0; ii < num && ptr != NULL; 
       ++ii, ptr = ptr->next);

  if (ptr == NULL)
    {
      ErrorMsg("BitBuffer", "find_pos", "Unexpected NULL ptr in buffer structure. ii =%d, num = %d\n", ii, num);
      return pos;
    }

  pos.nodeptr = ptr;

  // compute the word offset
  num = index % (BITS_PER_WORD*BB_NODE_SIZE);
  ii = num/BITS_PER_WORD;

  pos.word_idx = ii;

  // now compute the bit location
  num %= BITS_PER_WORD;

  pos.bit_idx = num;

  return pos;

}

//----------------------------------------------------------------------

int BitBuffer::position_read_ptr(int bit_pos)
{
  struct BufferPos pos;

  // find the position in question
  if (bit_counter < bit_pos)
    return -1;

  if (bit_pos == bit_counter)
    { // read ptr position is at end

     // written equality explicitly for IRIX compiler
      pos.nodeptr = buffer_end;
      pos.word_idx = in_word_index;
      pos.bit_idx = in_bit_index;
     
      // previously 
      //pos = (struct BufferPos) {buffer_end, in_word_index, in_bit_index};
    }
  else
    {
      pos = find_pos(bit_pos);
    }

  read_ptr = pos.nodeptr;
  read_word_index = pos.word_idx;
  read_bit_index = pos.bit_idx;

#if DEBUG
  printf("Pos read ptr: ptr 0x%X %d %d, start 0x%X %d %d\n",
	 read_ptr, read_word_index, read_bit_index, buffer_start,
	 out_word_index, out_bit_index);
#endif

  rbits_from_end = bit_counter - bit_pos;

  return 0;
}

//----------------------------------------------------------------------

int BitBuffer::read_bits(int numb_bits)
{
  int data = 0;
  int bits_read;

  if (numb_bits < 1 || numb_bits > 16)
    return -1;
  
  if ((bits_read = read_bits(numb_bits, data, false)) != numb_bits)
    return -1;

  return data;
}

//----------------------------------------------------------------------

/*
 * Bits are read from each packed word starting from the 
 * MSB and proceeding towards the LSB.
 * 
 *   MSB               LSB
 *    |_________________|
 *        time ->
 */


int BitBuffer::read_bits(int numb_bits, int& data, int get_available)
{
 
  unsigned long m = 0;
  unsigned long bmask = 0;
  int word_idx, bit_idx;
  int bits_read = 0;
  BufferNode *ptr;

  // initialize data to zero
  data = 0;

#if DEBUG
  printf("out_word_index %d, out_bit_index %d in_word_index %d in_bit_index %d\n",
	 out_word_index, out_bit_index, in_word_index, in_bit_index);
#endif

  if (numb_bits < 1 || numb_bits > (sizeof(int) << 3))
    return -1;

  if (preserve)
    {
      if (!get_available && (rbits_from_end < numb_bits))
	return -1;
      else
	numb_bits = Min(numb_bits, rbits_from_end);
      ptr = read_ptr;
      word_idx = read_word_index;
      bit_idx = read_bit_index;

      rbits_from_end -= numb_bits;
    }
  else
    {
      // determine if there are enough bits
      if (!get_available && (bit_counter < numb_bits))
	return -1;
      else
	numb_bits = Min(numb_bits, bit_counter);
      ptr = buffer_start;
      word_idx = out_word_index;
      bit_idx = out_bit_index;

      // just subtract all the bits from the 
      // bit counter now for efficiency
      bit_counter -= numb_bits;
      // if not preserve then readptr is at beginning and we need 
      // to subtract from rbits_from_end
      rbits_from_end -= numb_bits;
    }


  m = 1 << (numb_bits - 1);
  bmask = scInitBmask >> (bit_idx);
  
  while (numb_bits && (ptr != buffer_end))
    {
      if (bit_idx >= BITS_PER_WORD)
	{
	  bit_idx = 0;   bmask = scInitBmask;
	  if ((++word_idx) >= BB_NODE_SIZE)
	    {
	      BufferNode *tmp;
	      tmp = ptr->next;
	      if (!preserve)
		{ 
		  delete ptr;
		}
	      ptr = tmp;
	      word_idx = 0;
	      continue;
	    }
	}

      if (ptr->data[word_idx] & bmask)
	data |= m;
      m >>= 1;
      ++bit_idx; bmask >>= 1;
      --numb_bits;
      ++bits_read;
    }
 
  // bmask and bit_idx already set from above

  while (numb_bits)
    { // buffer_start == buffer_end here 
      
      if (bit_idx >= BITS_PER_WORD)
	{
	  bit_idx = 0; bmask = scInitBmask;
	  ++word_idx;
	  continue;
	}


      if (ptr->data[word_idx] & bmask)
	data |= m;
      m >>= 1;
      if (!preserve && (word_idx == in_word_index))
	{
	  --in_bit_index; ptr->data[word_idx] <<= 1;
	}
      else
	{
	  ++bit_idx; bmask >>= 1;
	}
      --numb_bits;
      ++bits_read;
    }

  // update location values
  if (preserve)
    {
      read_ptr = ptr;
      read_word_index = word_idx;
      read_bit_index = bit_idx;
    }
  else
    {
      buffer_start = ptr;
      out_word_index = word_idx;
      out_bit_index = bit_idx;
    }

  return bits_read;

}

//----------------------------------------------------------------------

int BitBuffer::operator[](int index)
{
  int ii, num;
  long word;
  BufferNode *ptr;
  struct BufferPos pos;
  
  pos = find_pos(index);

  ptr = pos.nodeptr;
  if (ptr == NULL)
    return -1;

  ii = pos.word_idx;

  num = pos.bit_idx;
  
  word = ptr->data[ii];

#if 0
  // This code should be removed soon.
  // I haven't thoroughly tested everything 
  // so I wanted to leave it for reference
  // in case of problems.
  if ((ptr == buffer_end && ii == in_word_index) && 
      (!((ptr == buffer_start) && (ii == out_word_index))))
    {
      num = (1 << (in_bit_index+num));
    }
  else
    num = (scInitBmask >> num);
#endif

  num = (scInitBmask >> num);

  return ((word & num) != 0);

}

//----------------------------------------------------------------------

int BitBuffer::dump_bits(FILE *fp, int start_pos, int end_pos, bool reverse)
{
  int cnt;
  int old_read_pos;
  int old_preserve_state;

  start_pos = (start_pos == BB_START) ? 0 : start_pos;
  end_pos = (end_pos == BB_END) ? (bit_counter-1): end_pos;
  // check that position values are valid
  
  if (start_pos < 0 || start_pos > end_pos)
    return BB_DUMP_ESPEC;
  
  else if (end_pos > bit_counter)
    return BB_DUMP_ESPEC;

  // save some of the old state info to be restored later
  old_preserve_state = preserve;
  old_read_pos = bit_counter - rbits_from_end;
  
  // set state to allow reading without deleting
  preserve = 1;
  // set read ptr to location of interest
  position_read_ptr(start_pos);


  cnt = end_pos - start_pos + 1;

  // now output the bits justifying those that don't fill 
  // a byte to the most significant bits.
  
  int word;
  int bytes = (cnt >> 3); // compute number of bytes
  for (; bytes; --bytes)
    {
      if (read_bits(8, word) != 8)
	return BB_DUMP_EFILE;
      if (reverse) reverse_byte(word);
      if (fputc(word, fp) == EOF)
	return BB_DUMP_EFILE;
    }
  cnt %= 8;
  if (cnt)
    {
      if (read_bits(cnt, word) != cnt)
	return BB_DUMP_EFILE;
      if (reverse)
	{ 
	  reverse_byte(word);
	  word >>= (8-cnt);
	}
      else
	{
	  word <<= (8-cnt);
	}
      if (fputc(word, fp) == EOF)
	return BB_DUMP_EFILE;
    }

  // restore buffer state
  preserve = old_preserve_state;
  position_read_ptr(old_read_pos);


  return BB_DUMP_SUCCESS;

}

//----------------------------------------------------------------------

int BitBuffer::fill_from_file(FILE* fp, int number_of_bytes, bool reverse)
{
  int word;


  if (ALL_BITS == number_of_bytes)
    {
      int cnt = 0;
      // read until an EOF
      while ((word = fgetc(fp)) != EOF)
	{
	  if (reverse) reverse_byte(word);
	  buffer_bits(8, word);
	  ++cnt;
	}

      return cnt; // return number of bytes
    }
  else
    {
      int bytes;
      int cnt = 0;
      
      for (bytes = number_of_bytes; bytes; --bytes)
	{
	  if ((word = fgetc(fp)) != EOF)
	    {
	      if (reverse) reverse_byte(word);
	      buffer_bits(8, word);
	      ++cnt;
	    }
	  else
	    break;
	}

      return cnt;

    }

}

//----------------------------------------------------------------------

int BitBuffer::print_bits(int start_pos, int end_pos,ostream& stream)
{
#define BYTES_PER_LINE 5
  int cnt;
  int word;
  int old_read_pos;
  int old_preserve_state;

  start_pos = (start_pos == BB_START) ? 0 : start_pos;
  end_pos = (end_pos == BB_END) ? (bit_counter-1) : end_pos;
  // check that position values are valid
  
  if (start_pos < 0 || start_pos > end_pos)
    return BB_DUMP_ESPEC;
  
  else if (end_pos > bit_counter)
    return BB_DUMP_ESPEC;

  // save some of the old state info to be restored later
  old_preserve_state = preserve;
  old_read_pos = bit_counter - rbits_from_end;
  
  // set state to allow reading without deleting
  preserve = 1;
  // set read ptr to location of interest
  position_read_ptr(start_pos);

  cnt = end_pos - start_pos + 1;

  stream << "Dumping " << cnt << " bits " << endl;

  unsigned int ii, jj, bits;
  for (ii = 0; cnt; ii = (ii+1)%BYTES_PER_LINE)
    {
      bits = read_bits(Min(cnt,8), word, true);
      for ( jj = (1 << (bits-1)); jj; jj>>= 1) 
	{
	  if ((jj & word) != 0) stream << "1";
	  else stream << "0";
	}
      cnt -= bits;
      if (ii == (BYTES_PER_LINE - 1))
	stream << endl;
    }
      
  // restore buffer state
  preserve = old_preserve_state;
  stream << endl << "Positioning read ptr to " << old_read_pos << endl;
  position_read_ptr(old_read_pos);


  return BB_DUMP_SUCCESS;

}

//----------------------------------------------------------------------

int BitBuffer::hamming_weight(int start_pos, int end_pos)
{
  // Return number of '1' bits or -1 on error

  int cnt;
  int old_read_pos;
  int old_preserve_state;

  start_pos = (start_pos == BB_START) ? 0 : start_pos;
  end_pos = (end_pos == BB_END) ? (bit_counter-1): end_pos;
  // check that position values are valid
  
  if (start_pos < 0 || start_pos > end_pos)
    return -1;
  
  else if (end_pos > bit_counter)
    return -1;

  // save some of the old state info to be restored later
  old_preserve_state = preserve;
  old_read_pos = bit_counter - rbits_from_end;
  
  // set state to allow reading without deleting
  preserve = 1;
  // set read ptr to location of interest
  position_read_ptr(start_pos);


  cnt = end_pos - start_pos + 1;


  // now compute the number of '1' bits
  
  int word, bits, jj;
  int weight = 0;
  while (cnt > 0)
    {
      bits = read_bits(Min(cnt,16), word, true);
      if (bits <= 0) break;
      for (jj = (1 << (bits-1)); jj; jj >>= 1)
	{
	  if (0 != (jj & word)) ++weight;
	}
      cnt -= bits;
    }

  // restore buffer state
  preserve = old_preserve_state;
  position_read_ptr(old_read_pos);


  return weight;

}

//----------------------------------------------------------------------

bool BitBuffer::concat(BitBuffer& src_buf)
{
  int old_preserve_state;
  int old_read_pos;

  // save src buffer state and turn on preserve mode
  old_read_pos = src_buf.bits_to_readptr();
  old_preserve_state = src_buf.set_preserve(1);

  int cnt = src_buf.bits_in_buffer(); // number of bits to add
  int bits;
  int word;
  while (cnt)
    {
      bits = src_buf.read_bits(Min(16,cnt), word);
      if (bits <= 0) break;
      buffer_bits(bits, word);
      cnt -= bits;
    }

  // restore state of src buffer
  src_buf.set_preserve(old_preserve_state);
  src_buf.position_read_ptr(old_read_pos);

  return (0 == cnt);

}

//----------------------------------------------------------------------


ostream& operator << (ostream& s, BitBuffer& inBuf)
{
  inBuf.print_bits(BB_START, BB_END, s);
  return s;
}

//----------------------------------------------------------------------

int BitBuffer::compare(BitBuffer& bufB)
{
  int bit_cnt;
  int diff = 0;

  bit_cnt = Min(bit_counter, bufB.bits_in_buffer());

  // put both into preserve mode and save old state
  int old_preserve_stateA;
  int old_read_posA;
  int old_preserve_stateB;
  int old_read_posB;

  old_read_posA = this->bits_to_readptr();
  old_preserve_stateA = this->set_preserve(1);
  old_read_posB = bufB.bits_to_readptr();
  old_preserve_stateB = bufB.set_preserve(1);

  int wordA;
  int wordB;
  int bitsA;
  int bitsB;
  while (bit_cnt)
    {
      if ((bitsA = this->read_bits(Min(bit_cnt,16),wordA)) <= 0)
	break;
      if ((bitsB = bufB.read_bits(Min(bit_cnt,16),wordB)) <= 0)
	break;
      if (bitsA != bitsB)
	return -1;
      else if (wordA != wordB)
	{
	  int jj;
	  wordA ^= wordB;
	  for (jj = (1 << (bitsA-1)); jj; jj >>= 1)
	    {
	      if (0 != (jj & wordA)) ++diff;
	    }
	}
      bit_cnt -= bitsA;
    }

  
  // restore the old buffer states
  this->set_preserve(old_preserve_stateA);
  this->position_read_ptr(old_read_posA);
  bufB.set_preserve(old_preserve_stateB);
  bufB.position_read_ptr(old_read_posB);

  return diff;
}


//----------------------------------------------------------------------

bool operator <= (BitBuffer& bufA, BitBuffer& bufB)
{
  if (bufA.bits_in_buffer() > bufB.bits_in_buffer())
    return false;

  int cnt = bufA.bits_in_buffer();

  // put both into preserve mode and save old state
  int old_preserve_stateA;
  int old_read_posA;
  int old_preserve_stateB;
  int old_read_posB;

  old_read_posA = bufA.bits_to_readptr();
  old_preserve_stateA = bufA.set_preserve(1);
  old_read_posB = bufB.bits_to_readptr();
  old_preserve_stateB = bufB.set_preserve(1);

  int wordA;
  int wordB;
  int bitsA;
  int bitsB;
  while (cnt)
    {
      if ((bitsA = bufA.read_bits(Min(cnt,16),wordA)) <= 0)
	{
	  break;
	}
      if ((bitsB = bufB.read_bits(Min(cnt,16),wordB)) <= 0)
	{
	  break;
	}
      if ((bitsA != bitsB) ||
	  (wordA != wordB))
	{
	  break;
	}
      cnt -= bitsA;
    }

  // restore the old buffer states
  bufA.set_preserve(old_preserve_stateA);
  bufA.position_read_ptr(old_read_posA);
  bufB.set_preserve(old_preserve_stateB);
  bufB.position_read_ptr(old_read_posB);

  return (cnt == 0);
}

//----------------------------------------------------------------------

bool BitBuffer::copy_range(BitBuffer& src_buf, 
			   int start_pos, 
			   int end_pos)
{
  int old_read_pos;
  int old_preserve_state;
  int cnt;

  start_pos = (start_pos == BB_START) ? 0 : start_pos;
  end_pos = (end_pos == BB_END) ? (src_buf.bit_counter-1) : end_pos;
  // check that position values are valid
  
  if (start_pos < 0 || start_pos > end_pos)
    return false;
  
  else if (end_pos > src_buf.bit_counter)
    return false;

  // save some of the old state info to be restored later
  old_preserve_state = src_buf.preserve;
  old_read_pos = src_buf.bit_counter - src_buf.rbits_from_end;
  
  // set state to allow reading without deleting
  src_buf.preserve = 1;
  // set read ptr to location of interest
  src_buf.position_read_ptr(start_pos);

  cnt = end_pos - start_pos + 1;

  int bits;
  int word;
  
  while (cnt)
    {
      bits = src_buf.read_bits(Min(cnt,16), word, true);
      if (bits <= 0) break;
      buffer_bits(bits, word);
      cnt -= bits;
    }

  // restore buffer state
  src_buf.preserve = old_preserve_state;
  src_buf.position_read_ptr(old_read_pos);
  
  return (0 == cnt);
}


//----------------------------------------------------------------------

int move_bits(BitBuffer *dest, BitBuffer *src, int num_bits)
{
  int ii, jj;
  int data;

  if (num_bits == ALL_BITS)
    num_bits = src->rbits_to_end();
  else
    num_bits = Min(num_bits, src->rbits_to_end());

  for (ii = num_bits; ii > 0; )
    {
      jj = Min(ii, 16);
      src->read_bits(jj, data);
      dest->buffer_bits(jj, data);
      ii -= jj;
    }

  return num_bits;
}

//----------------------------------------------------------------------


/* 
 * This routine reverses the LSByte 
 * of the word.  This is just here to 
 * support reading/files stored in Said/Pearlman
 * format
 *
 */
static void reverse_byte(int& word)
{
  int m;
  int r;
  int tmp = word;
  word = 0; 
  for (m = (1<<7), r = 1; m ; m >>= 1, r <<= 1)
    {
      if ((tmp & m) != 0)
	{
	  word |= r;
	}
    }

}


//----------------------------------------------------------------------

bool BitBuffer::set_bit(int bit_index, int bit_value)
{
  int ii, num;
  BufferNode *ptr;
  struct BufferPos pos;

  // find position of the bit in question
  pos = find_pos(bit_index);

  ptr = pos.nodeptr;
  if (ptr == NULL)
    return false;

  ii = pos.word_idx;

  num = pos.bit_idx;
  
  if (bit_value)
    {
      num = (scInitBmask >> num);
      ptr->data[ii] |= num;
    }
  else 
    {
      num = ~(scInitBmask >> num);
      ptr->data[ii] &= num;
    }

  return true;

}

//----------------------------------------------------------------------

int BitBuffer::h_xor(BitBuffer& xor_buf)
{
  int bit;
  int cnt = Min(bit_counter, xor_buf.bits_in_buffer());
  int ii;
  int xor_preserve_state;
  int xor_read_pos;
  int old_preserve_state;
  int old_read_pos;


  // save the readptr and preserve states
  xor_preserve_state = xor_buf.preserve;
  xor_read_pos = xor_buf.bit_counter - xor_buf.rbits_from_end;

  xor_buf.preserve = 1;
  xor_buf.position_read_ptr(0);
  
  old_preserve_state = preserve;
  old_read_pos = bits_to_readptr();
  preserve = 1;
  position_read_ptr(0);

  for (ii = 0; ii < cnt; ++ii)
    { // loop over bits in question
      if (xor_buf.read_bits(1, bit) != 1)
	return ii;

      // read the bit at read ptr
      if (bit)
	{
	  int jj, num;
	  BufferNode *ptr;
	  struct BufferPos pos;
	  
	  pos = find_pos(ii);
	  
	  ptr = pos.nodeptr;
	  if (ptr == NULL)
	    return ii;

	  jj = pos.word_idx;
	  num = pos.bit_idx;
	  num = (scInitBmask >> num);

	  ptr->data[jj] ^= num;
	  
	}

      
    }

  xor_buf.preserve = xor_preserve_state;
  xor_buf.position_read_ptr(xor_read_pos);

  preserve = old_preserve_state;
  position_read_ptr(old_read_pos);


  return ii;
}
