/*
 * mcn_streaming sender
 * Copyright (c) 2010 Qin Chen <eric.qin.chen@gmail.com>
 *
 * This file is part of mcn_streaming.
 *
 * mcn_streaming is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * mcn_streaming is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with mcn_streaming; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */


/*
 * Sender example setup
 * sends the output of v4l2src as h264 encoded RTP on port 5000, RTCP is sent on
 * port 5001. The destination is 127.0.0.1.
 * the video receiver RTCP reports are received on port 5005
 *
 * .-------.    .-------.    .-------.      .----------.     .-------.
 * |v4l2   |    |x264enc|    |h264pay|      | rtpbin   |     |udpsink|  RTP
 * |      src->sink    src->sink    src->send_rtp send_rtp->sink     | port=5000
 * '-------'    '-------'    '-------'      |          |     '-------'
 *                                          |          |
 *                                          |          |     .-------.
 *                                          |          |     |udpsink|  RTCP
 *                                          |    send_rtcp->sink     | port=5001
 *              .-------.    .--------.     |          |     '-------' sync=false
 *    RTCP      |udpsrc |    |identity|     |          |               async=false
 *    port=5005 |      src->sink     src->recv_rtcp    |
 *              '-------'    '--------'     '----------'
 */


#include <config.h>
#include <stdio.h>
#include <gst/gst.h>
#include <glib.h>
#include <gst/rtp/gstrtcpbuffer.h>
#include <gst/controller/gstcontroller.h>

// video source
static char *vsrc = NULL;
static int width  = 176;
static int height = 144;
static int fps    = 15;
static int bitrate = 128;

// video encoder
static char *vencoder = NULL;
static gboolean h263enc = FALSE;

// rtp encoder
static char *rtpencoder = NULL;
static int mtu_size = 1024;

// host and ports
static char *host = NULL;
static int rtp_port    = 5000;
static int rtcp_port   = 5001;
static int rtcp_port_r = 5005; // port for receiving receiver RTCP packets


static gboolean bus_call(GstBus *bus, GstMessage *msg, gpointer data) {
	GMainLoop *loop = (GMainLoop *) data;

	switch (GST_MESSAGE_TYPE (msg)) {

	case GST_MESSAGE_EOS:
		g_print("End of stream\n");
		g_main_loop_quit(loop);
		break;

	case GST_MESSAGE_ERROR: {
		gchar *debug;
		GError *error;

		gst_message_parse_error(msg, &error, &debug);
		g_free(debug);

		g_printerr("Error: %s\n", error->message);
		g_error_free(error);

		g_main_loop_quit(loop);
		break;
	}
	default:
		break;
	}

	return TRUE;
}

static GOptionEntry entries[] = {
		{ "vsrc", 's', 0, G_OPTION_ARG_STRING, &vsrc,
				"Specify video source", NULL },
		{ "width", 'w', 0, G_OPTION_ARG_INT, &width,
				"Specify video width", NULL },
		{ "height", 'h', 0, G_OPTION_ARG_INT, &height,
				"Specify video height", NULL },
		{ "fps", 'f', 0, G_OPTION_ARG_INT, &fps,
				"Specify video frame rate", NULL },
		{ "bitrate", 'b', 0, G_OPTION_ARG_INT, &bitrate,
				"Specify video bit rate", NULL },
		{ "venc", 'e', 0, G_OPTION_ARG_STRING, &vencoder,
				"Specify video encoder", NULL },
		{ "host", 'd', 0, G_OPTION_ARG_STRING, &host,
				"Specify receiver IP", NULL },
		{ "rtp-port", 't', 0, G_OPTION_ARG_INT, &rtp_port,
				"Specify receiver port for RTP", NULL },
		{ "rtcp-port-tx", 'x', 0, G_OPTION_ARG_INT, &rtcp_port,
				"Specify receiver port for RTCP", NULL },
		{ "rtcp-port-rx", 'r', 0, G_OPTION_ARG_INT, &rtcp_port_r,
				"Specify sender port for RTCP", NULL },
		{ "mtu-size", 'm', 0, G_OPTION_ARG_INT, &mtu_size,
				"Specify maximum size of one RTP packet", NULL },
		{ NULL }
};

static
void parse_options(int c, char **v){
	  GOptionContext *ctx;
	  GError *err = NULL;
	  GOptionGroup *group;

	  /* we must initialize the threading system before using any
	   * other GLib function, such as g_option_context_new() */
	  if (!g_thread_supported ())
	    g_thread_init (NULL);

	  ctx = g_option_context_new ("- mcn_streaming sender");
	  //g_option_context_add_main_entries (ctx, entries, NULL);

	  group = g_option_group_new("sender", "mcn_streaming Sender Options:",
			  "Show mcn_streaming sender options", NULL, NULL);
	  g_option_group_add_entries(group, entries);
	  g_option_context_add_group(ctx, group);

	  //Need this for gst option initialization
	  g_option_context_add_group (ctx, gst_init_get_option_group ());
	  if (!g_option_context_parse (ctx, &c, &v, &err)) {
	    g_print ("Failed to initialize: %s\n", err->message);
	    g_error_free (err);
	  }

	  //Default video source to v4l2src
	  if ( !vsrc ) {
		  vsrc = g_strdup("v4l2src");
	  }

	  //Default video encoder to H.264
	  if ( !vencoder ) {
		  vencoder = g_strdup("x264enc");
	  }

	  //Default receiver IP to localhost
	  if ( !host ) {
		  host = g_strdup("127.0.0.1");
	  }

}

//Select RTP encoder according to video codec
static
char* select_rtp_encoder(char *vcodecname)
{
	char *rtpenc;
	if ( g_strcmp0(vcodecname, "ffenc_h263") == 0 ||
			g_strcmp0(vcodecname, "ffenc_h263p") == 0) {
		rtpenc = "rtph263ppay";
		h263enc = TRUE;
	}
//	else if (g_strcmp0(vcodecname, "ffenc_mpeg4") == 0) {
//		rtpenc = "rtpmp4vpay";
//	}
//	else if (g_strcmp0(vcodecname, "theoraenc") == 0) {
//		rtpenc = "rtptheorapay";
//	}
	else {
		//Default to H.264
		rtpenc = "rtph264pay";
	}

	//g_debug ("RTP encoder %s was created\n", rtpenc);
	return rtpenc;
}

typedef struct _rtcp_info rtcp_info;
struct _rtcp_info
{
	guint32 jitter;
	guint8  fractionlost;
	gint32  packetslost;
};

//parse RTCP packets
static gboolean
process_rtcp_packet(GstRTCPPacket *packet, rtcp_info *rtcp_pkt){
	guint32 ssrc, rtptime, packet_count, octet_count;
	guint64 ntptime;
	guint count, i;

	count = gst_rtcp_packet_get_rb_count(packet);
	g_debug("    count         %d", count);
	for (i=0; i<count; i++) {
		guint32 exthighestseq, jitter, lsr, dlsr;
		guint8 fractionlost;
		gint32 packetslost;

		gst_rtcp_packet_get_rb(packet, i, &ssrc, &fractionlost,
				&packetslost, &exthighestseq, &jitter, &lsr, &dlsr);

		g_debug("    block         %d", i);
		g_debug("    ssrc          %d", ssrc);
		g_debug("    highest   seq %d", exthighestseq);
		g_debug("    jitter        %d", jitter);
		g_debug("    fraction lost %d", fractionlost);
		g_debug("    packet   lost %d", packetslost);

		rtcp_pkt->fractionlost = fractionlost;
		rtcp_pkt->jitter = jitter;
		rtcp_pkt->packetslost = packetslost;
	}

	//g_debug("Received rtcp packet");

	return TRUE;
}

//create and send upstream event to video encoder
static gboolean
send_event_to_encoder(GstElement *venc, rtcp_info *rtcp_pkt)
{
	GstPad *pad;
	GstEvent *event;
	GstStructure *structure;

	g_debug("Send event to %s", gst_element_get_name(venc));

	//make custom upstream event
	structure = gst_structure_new("rtcp_event",
			"type", G_TYPE_STRING, "receiver_report",
			"jitter", G_TYPE_UINT, rtcp_pkt->jitter,
			"frac_loss", G_TYPE_UINT, rtcp_pkt->fractionlost,
			"pkt_loss", G_TYPE_INT, rtcp_pkt->packetslost,
			NULL);
	if (!structure) {
		g_printerr("Failed to create event structure\n");
		return 0;
	}

	event = gst_event_new_custom (GST_EVENT_CUSTOM_UPSTREAM, structure);
	if (!event) {
		g_printerr("Failed to create event\n");
		return 0;
	}

	//get venc src pad
    pad = gst_element_get_static_pad (venc, "src");
	if (!pad) {
		g_printerr("Failed to get encoder source pad\n");
		return 0;
	}

	//send event to venc src pad
    gst_pad_send_event(pad, event);

    gst_object_unref (pad);
}

//handle RTCP packet
static gboolean
cb_receive_rtcp(GstElement *src, GstBuffer *buf, gpointer data){
	GstRTCPPacket packet;
	gboolean more;
	GstElement *venc;
	rtcp_info rtcp_pkt;

	if (!gst_rtcp_buffer_validate(buf))
		g_debug("Received invalid RTCP packet");

	g_debug("Received rtcp packet");

	venc = (GstElement *)(data);

	buf = gst_buffer_make_metadata_writable(buf);
	more = gst_rtcp_buffer_get_first_packet(buf, &packet);
	while (more) {
		GstRTCPType type;

		type = gst_rtcp_packet_get_type(&packet);
		switch (type) {
		case GST_RTCP_TYPE_RR:
			process_rtcp_packet(&packet, &rtcp_pkt);
			//g_debug("RR");
			send_event_to_encoder(venc, &rtcp_pkt);
			break;
		default:
			g_debug("Other types");
			break;
		}
		more = gst_rtcp_packet_move_to_next(&packet);
	}

	//gst_buffer_unref(buf);
	return TRUE;
}

static
GstElement* construct_sender_pipeline(void){
	GstElement *pipeline, *source, *time, *gstrtpbin, *venc, *rtpenc;
	GstElement *udpsink_rtp, *udpsink_rtcp, *udpsrc_rtcp, *identity;
	GstCaps *caps;
	GstPad  *pad;
	//GstController *controller;
	gboolean err;
	GstPadLinkReturn res;

	/* Create gstreamer pipeline */
	pipeline = gst_pipeline_new("mcn_sender");
	if ( !pipeline ) {
		g_printerr("Failed to create pipeline\n");
		return 0;
	}

	//Create gstrtpbin
	gstrtpbin = gst_element_factory_make("gstrtpbin", "rtpbin");
	if ( !gstrtpbin ) {
		g_printerr("Failed to create gstrtpbin\n");
		return 0;
	}

	//Video source initialization
	source = gst_element_factory_make(vsrc, "video-source");
	if ( !source ) {
		g_printerr("Failed to create %s\n", vsrc);
		return 0;
	}
	g_free(vsrc);

	//Time overlay
	time = gst_element_factory_make("timeoverlay", "time-overlay");
	if ( !time ) {
		g_printerr("Failed to create timeoverlay\n");
		return 0;
	}

	//Create video encoder
	venc = gst_element_factory_make(vencoder, "video-encoder");
	if ( !venc ) {
		g_printerr("Failed to create %s\n", vencoder);
		return 0;
	}
	//kbits/sec --> bits/sec for H.264 encoder
	if (g_strcmp0(vencoder, "x264enc") != 0) {
		bitrate *= 1024;
	}
	g_object_set(G_OBJECT (venc), "bitrate", bitrate, NULL);
	//bitrate is not controllable
	//controller = gst_object_control_properties(G_OBJECT (venc), "bitrate", NULL);

	//Choose RTP encoder according to video codec
	rtpencoder = g_strdup(select_rtp_encoder(vencoder));
	g_free(vencoder);

	//Create rtp encoder
	rtpenc = gst_element_factory_make(rtpencoder, "rtp-encoder");
	if ( !rtpenc ) {
		g_printerr("Failed to create %s\n", rtpencoder);
		return 0;
	}
	g_object_set(G_OBJECT (rtpenc), "mtu", mtu_size, NULL);
#if 0
	if (h263enc == TRUE) {
		g_object_set(G_OBJEC T (rtpenc), "caps",
			gst_caps_from_string("encoding-name=(string)H263-2000"), NULL);
	}
#endif //0

	//Create udp sink for RTP
	udpsink_rtp = gst_element_factory_make("udpsink", "udpsink-rtp");
	if ( !udpsink_rtp ) {
		g_printerr("Failed to create udpsink\n");
		return 0;
	}
	g_object_set(G_OBJECT (udpsink_rtp), "host", host, "port", rtp_port, NULL);

	//Create udp sink for RTCP
	udpsink_rtcp = gst_element_factory_make("udpsink", "udpsink-rtcp");
	if ( !udpsink_rtcp ) {
		g_printerr("Failed to create udpsink\n");
		return 0;
	}
	g_object_set(G_OBJECT (udpsink_rtcp), "host", host, "port", rtcp_port,
			"sync",FALSE, "async",FALSE, NULL);
	g_free(host);

	//RTCP source initialization
	udpsrc_rtcp = gst_element_factory_make("udpsrc", "udpsrc-rtcp");
	if ( !udpsrc_rtcp ) {
		g_printerr("Failed to create udpsrc\n");
		return 0;
	}
	g_object_set(G_OBJECT (udpsrc_rtcp), "port", rtcp_port_r, NULL);
	g_object_set(G_OBJECT (udpsrc_rtcp), "caps",
			gst_caps_from_string("application/x-rtcp"), NULL);

	//identity element connected to updsrc_rtcp, to check if received any packets or not
	identity = gst_element_factory_make("identity", "udpsrc-rtcp-identity");
	if ( !identity ) {
		g_printerr("Failed to create identity element\n");
		return 0;
	}
	//set identity element to sync to clock
	g_object_set(G_OBJECT (identity), "sync", TRUE, NULL);
	//g_object_set(G_OBJECT (identity), "dump", TRUE, NULL);
    //g_signal_connect (identity, "handoff", G_CALLBACK (cb_receive_rtcp), NULL);
	//probe RTCP packets
    pad = gst_element_get_static_pad (identity, "sink");
    gst_pad_add_buffer_probe (pad, G_CALLBACK (cb_receive_rtcp), venc);
    gst_object_unref (pad);


	/* Set up the pipeline */
	gst_bin_add_many(GST_BIN (pipeline), source, time, venc,
			rtpenc, gstrtpbin, udpsink_rtp, udpsink_rtcp, udpsrc_rtcp, identity, NULL);

	//Set up the video encoding parameters
	caps = gst_caps_new_simple ("video/x-raw-yuv",
			"width",  G_TYPE_INT, width,
			"height", G_TYPE_INT, height,
			"framerate", GST_TYPE_FRACTION, fps, 1, NULL);
	if ( !caps ) {
		g_printerr("Failed to create caps\n");
		return 0;
	}
	err = gst_element_link_filtered(source, time, caps);
	gst_caps_unref(caps);
	if ( err==FALSE ) {
		g_printerr("Failed to link %s and timeoverlay\n", vsrc);
		return 0;
	}

	err = gst_element_link_many(time, venc, rtpenc, NULL);
	if ( err==FALSE ) {
		g_printerr("Failed to link elements\n");
		return 0;
	}

	//link rtp sender to gstrtpbin
    pad = gst_element_get_request_pad(gstrtpbin, "send_rtp_sink_%d");
	if ( !pad ) {
		g_printerr("Failed to create pad\n");
		return 0;
	}
    res = gst_pad_link(gst_element_get_pad(rtpenc, "src"), pad);
    if ( GST_PAD_LINK_FAILED(res)  ) {
		g_printerr("Failed to link pads\n");
		return 0;
	}
    err = gst_element_link_many(gstrtpbin, udpsink_rtp, NULL);
	if ( err==FALSE ) {
		g_printerr("Failed to link elements\n");
		return 0;
	}
    gst_object_unref(GST_OBJECT (pad));

	//link rctp sender to gstrtpbin
    pad = gst_element_get_request_pad(gstrtpbin, "send_rtcp_src_%d");
	if ( !pad ) {
		g_printerr("Failed to create pad\n");
		return 0;
	}
    err = gst_element_link_many(gstrtpbin, udpsink_rtcp, NULL);
	if ( err==FALSE ) {
		g_printerr("Failed to link elements\n");
		return 0;
	}
    gst_object_unref(GST_OBJECT (pad));

	//link rctp receiver to gstrtpbin
#if 0
    pad = gst_element_get_request_pad(gstrtpbin, "recv_rtcp_sink_%d");
 	if ( !pad ) {
		g_printerr("Failed to create pad\n");
		return 0;
	}
    res = gst_pad_link(gst_element_get_pad(udpsrc_rtcp, "src"), pad);
    if ( GST_PAD_LINK_FAILED(res)  ) {
		g_printerr("Failed to link pads\n");
		return 0;
	}
    gst_object_unref(GST_OBJECT (pad));
#endif //0
    pad = gst_element_get_request_pad(gstrtpbin, "recv_rtcp_sink_%d");
 	if ( !pad ) {
		g_printerr("Failed to create pad\n");
		return 0;
	}
    err = gst_element_link_many(udpsrc_rtcp, identity, NULL);
	if ( err==FALSE ) {
		g_printerr("Failed to link elements\n");
		return 0;
	}
    res = gst_pad_link(gst_element_get_pad(identity, "src"), pad);
    if ( GST_PAD_LINK_FAILED(res)  ) {
		g_printerr("Failed to link pads\n");
		return 0;
	}
    gst_object_unref(GST_OBJECT (pad));

	return pipeline;
}

int main(int argc, char **argv) {
	const gchar *nano_str;
	guint major, minor, micro, nano;
	GMainLoop  *loop;
	GstElement *pipeline;
	GstBus  *bus;

	//have to put parse_options before gst_init
	//to show mcn_streaming help instead of GStreamer help
	parse_options(argc, argv);

	gst_init(&argc, &argv);

	//for dynamic controllable parameters
	gst_controller_init (&argc, &argv);

	gst_version(&major, &minor, &micro, &nano);
	if (nano == 1)
		nano_str = "(CVS)";
	else if (nano == 2)
		nano_str = "(Prerelease)";
	else
		nano_str = "";
	g_print("This is " PACKAGE_STRING "\n");
	g_print("This program is linked against GStreamer %d.%d.%d %s\n\n", major,
			minor, micro, nano_str);

	loop = g_main_loop_new(NULL, FALSE);

	//Construct sender pipeline
	pipeline = construct_sender_pipeline();
	if ( !pipeline ) {
		g_printerr("Failed to construct pipeline\n");
		return -1;
	}

	/* we add a message handler */
	bus = gst_pipeline_get_bus(GST_PIPELINE (pipeline));
	gst_bus_add_watch(bus, bus_call, loop);
	gst_object_unref(bus);
	//g_signal_connect (demuxer, "pad-added", G_CALLBACK (on_pad_added), decoder);

	/* Set the pipeline to "playing" state*/
	g_print("Now streaming...\n");
	gst_element_set_state(pipeline, GST_STATE_PLAYING);

	/* Iterate */
	g_print("Running...\n");
	g_main_loop_run(loop);

	/* Out of the main loop, clean up nicely */
	g_print("Returned, stopping streaming\n");
	gst_element_set_state(pipeline, GST_STATE_NULL);

	g_print("Deleting pipeline\n");
	gst_object_unref(GST_OBJECT (pipeline));

	return 0;
}

