/*
 * mcn_streaming receiver
 * Copyright (c) 2010 Qin Chen <eric.qin.chen@gmail.com>
 *
 * This file is part of mcn_streaming.
 *
 * mcn_streaming is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * mcn_streaming is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with mcn_streaming; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * Receiver setup
 *
 *  receives H264 encoded RTP video on port 5000, RTCP is received on  port 5001.
 *  the receiver RTCP reports are sent to port 5005
 *
 *             .-------.      .----------.     .---------.   .-------.   .-----------.
 *  RTP        |udpsrc |      | rtpbin   |     |h264depay|   |h264dec|   |xvimagesink|
 *  port=5000  |      src->recv_rtp recv_rtp->sink     src->sink   src->sink         |
 *             '-------'      |          |     '---------'   '-------'   '-----------'
 *                            |          |
 *                            |          |     .-------.
 *                            |          |     |udpsink|  RTCP
 *                            |    send_rtcp->sink     | port=5005
 *             .-------.      |          |     '-------' sync=false
 *  RTCP       |udpsrc |      |          |               async=false
 *  port=5001  |     src->recv_rtcp      |
 *             '-------'      '----------'
 */

#include <config.h>
#include <stdio.h>
#include <gst/gst.h>
#include <glib.h>


// video decoder
static char *vdecoder = NULL;
static gboolean h263dec = FALSE;

// rtp decoder
static char *rtpdecoder = NULL;

// jitter buffer latency
// 500ms seems to be a good tradeoff between smoothness and latency
static int jitter_latency = 500;

// host and ports
static char *host = NULL;
static int rtp_port    = 5000;
static int rtcp_port   = 5001; // port for receiving sender RTCP packets
static int rtcp_port_s = 5005; // port for sending receiver RTCP packets


static gboolean
bus_call (GstBus     *bus,
          GstMessage *msg,
          gpointer    data)
{
  GMainLoop *loop = (GMainLoop *) data;

  switch (GST_MESSAGE_TYPE (msg)) {

    case GST_MESSAGE_EOS:
      g_print ("End of stream\n");
      g_main_loop_quit (loop);
      break;

    case GST_MESSAGE_ERROR: {
      gchar  *debug;
      GError *error;

      gst_message_parse_error (msg, &error, &debug);
      g_free (debug);

      g_printerr ("Error: %s\n", error->message);
      g_error_free (error);

      g_main_loop_quit (loop);
      break;
    }
    default:
      break;
  }

  return TRUE;
}

static GOptionEntry entries[] = {
		{ "vdec", 'd', 0, G_OPTION_ARG_STRING, &vdecoder,
				"Specify video decoder", NULL },
		{ "host", 's', 0, G_OPTION_ARG_STRING, &host,
				"Specify sender IP", NULL },
		{ "rtp-port", 't', 0, G_OPTION_ARG_INT, &rtp_port,
				"Specify port for RTP", NULL },
		{ "rtcp-port-rx", 'x', 0, G_OPTION_ARG_INT, &rtcp_port,
				"Specify port for RTCP", NULL },
		{ "rtcp-port-tx", 'r', 0, G_OPTION_ARG_INT, &rtcp_port_s,
				"Specify sender port for RTCP", NULL },
		{ "latency", 'l', 0, G_OPTION_ARG_INT, &jitter_latency,
				"Specify jitter buffer latency", NULL },
		{ NULL }
};

static
void parse_options(int c, char **v){
	  GOptionContext *ctx;
	  GError *err = NULL;
	  GOptionGroup *group;

	  /* we must initialize the threading system before using any
	   * other GLib function, such as g_option_context_new() */
	  if (!g_thread_supported ())
	    g_thread_init (NULL);

	  ctx = g_option_context_new ("- mcn_streaming receiver");
	  //g_option_context_add_main_entries (ctx, entries, NULL);

	  group = g_option_group_new("receiver", "mcn_streaming Receiver Options:",
			  "Show mcn_streaming receiver options", NULL, NULL);
	  g_option_group_add_entries(group, entries);
	  g_option_context_add_group(ctx, group);

	  //Need this for gst option initialization
	  g_option_context_add_group (ctx, gst_init_get_option_group ());
	  if (!g_option_context_parse (ctx, &c, &v, &err)) {
	    g_print ("Failed to initialize: %s\n", err->message);
	    g_error_free (err);
	  }

	  //Default video decoder to H.264
	  if ( !vdecoder ) {
		  vdecoder = g_strdup("ffdec_h264");
	  }

	  //Default receiver IP to localhost
	  if ( !host ) {
		  host = g_strdup("127.0.0.1");
	  }

	  //Set flag if H.263 stream
	  if ( g_strcmp0(vdecoder, "ffdec_h263") == 0 ) {
		  h263dec = TRUE;
		  //g_debug("H263 stream\n");
	  }
}


//Select RTP encoder according to video codec
static
char* select_rtp_decoder(char *vcodecname)
{
	char *rtpdec;
	if ( g_strcmp0(vcodecname, "ffdec_h263") == 0 ) {
		rtpdec = "rtph263pdepay";
	}
//	else if (g_strcmp0(vcodecname, "ffdec_mpeg4") == 0) {
//		rtpdec = "rtpmp4vdepay";
//	}
//	else if (g_strcmp0(vcodecname, "theoradec") == 0) {
//		rtpdec = "rtptheoradepay";
//	}
	else {
		//Default to H.264
		rtpdec = "rtph264depay";
	}

	//g_debug ("RTP decoder %s was created\n", rtpdec);
	return rtpdec;
}


//recv_rtp_src_%d_%d_%d pad of gstrtpbin created
//this pad is a "sometimes" pad and will be created dynamically
//it cannot be requested or statically obtained
//after it is created, link it to sink pad of rtpdec
static void
cb_new_rtp_recv_src_pad (GstElement *element,
		GstPad     *pad,
		gpointer    data)
{
	gchar *name;
	GstElement *rtpdec = (GstElement *) data;
	GstPad *sinkpad;

	name = gst_pad_get_name (pad);
	//g_debug ("A new pad %s was created\n", name);
	g_free (name);

	sinkpad = gst_element_get_static_pad (rtpdec, "sink");
	gst_pad_link (pad, sinkpad);
	gst_object_unref (sinkpad);

}

static void
cb_receive_rtcp(GstElement *src, GstBuffer *buf, gpointer data){
	//g_debug("Received rtcp packet");
}

static
GstElement* construct_receiver_pipeline(void){
	GstElement *pipeline, *gstrtpbin, *vdec, *rtpdec, *vconv, *vsink;
	GstElement *udpsink_rtcp, *udpsrc_rtp, *udpsrc_rtcp, *identity;
	GstCaps *caps;
	GstPad  *pad;
	gboolean err;
	GstPadLinkReturn res;

	/* Create gstreamer pipeline */
	pipeline = gst_pipeline_new("mcn_receiver");
	if ( !pipeline ) {
		g_printerr("Failed to create pipeline\n");
		return 0;
	}

	//Create gstrtpbin
	gstrtpbin = gst_element_factory_make("gstrtpbin", "rtpbin");
	if ( !gstrtpbin ) {
		g_printerr("Failed to create gstrtpbin\n");
		return 0;
	}
	g_object_set(G_OBJECT (gstrtpbin), "latency", jitter_latency, NULL);

	//RTP ource initialization
	udpsrc_rtp = gst_element_factory_make("udpsrc", "udpsrc-rtp");
	if ( !udpsrc_rtp ) {
		g_printerr("Failed to create udpsrc\n");
		return 0;
	}
	g_object_set(G_OBJECT (udpsrc_rtp), "port", rtp_port, NULL);
	//gst_caps_new_simple and gst_element_linked_filter don't work
	if (g_strcmp0(vdecoder, "ffdec_h263") == 0) {
		//g_debug("H263 stream\n");
		g_object_set(G_OBJECT (udpsrc_rtp), "caps",
			gst_caps_from_string("application/x-rtp, clock-rate=(int)90000,"
					"media=(string)video,encoding-name=(string)H263-1998"), NULL);
//	} else if (g_strcmp0(vdecoder, "theoradec") == 0) {
//		g_object_set(G_OBJECT (udpsrc_rtp), "caps",
//			gst_caps_from_string("application/x-rtp, clock-rate=(int)90000,"
//					"media=(string)video,"
//					"delivery-method=(string)inline,encoding-name=(string)THEORA"), NULL);
	} else {
		g_object_set(G_OBJECT (udpsrc_rtp), "caps",
				gst_caps_from_string("application/x-rtp, clock-rate=(int)90000"), NULL);
	}

	//RTCP source initialization
	udpsrc_rtcp = gst_element_factory_make("udpsrc", "udpsrc-rtcp");
	if ( !udpsrc_rtcp ) {
		g_printerr("Failed to create udpsrc\n");
		return 0;
	}
	g_object_set(G_OBJECT (udpsrc_rtcp), "port", rtcp_port, NULL);
	g_object_set(G_OBJECT (udpsrc_rtcp), "caps",
			gst_caps_from_string("application/x-rtcp"), NULL);

	//identity element connected to updsrc_rtcp, to check if received any packets or not
	identity = gst_element_factory_make("identity", "udpsrc-rtcp-identity");
	if ( !identity ) {
		g_printerr("Failed to create identity element\n");
		return 0;
	}
	//set identity element to sync to clock
	g_object_set(G_OBJECT (identity), "sync", TRUE, NULL);
    g_signal_connect (identity, "handoff", G_CALLBACK (cb_receive_rtcp), NULL);

	//RTCP sink initialization
	udpsink_rtcp = gst_element_factory_make("udpsink", "udpsink-rtcp");
	if ( !udpsink_rtcp ) {
		g_printerr("Failed to create udpsink\n");
		return 0;
	}
	g_object_set(G_OBJECT (udpsink_rtcp), "host", host, "port", rtcp_port_s,
			"sync",FALSE, "async",FALSE, NULL);

	//Create video decoder
	vdec = gst_element_factory_make(vdecoder, "video-decoder");
	if ( !vdec ) {
		g_printerr("Failed to create %s\n", vdecoder);
		return 0;
	}

	//Choose RTP decoder according to video codec
	rtpdecoder = g_strdup(select_rtp_decoder(vdecoder));
	g_free(vdecoder);

	//Create rtp decoder
	rtpdec = gst_element_factory_make(rtpdecoder, "rtp-decoder");
	if ( !rtpdec ) {
		g_printerr("Failed to create %s\n", rtpdecoder);
		return 0;
	}

	//Create video converter
	vconv = gst_element_factory_make("ffmpegcolorspace", "video-converter");
	if ( !vconv ) {
		g_printerr("Failed to create ffmpegcolorspace\n");
		return 0;
	}

	//Create video sink
	vsink = gst_element_factory_make("xvimagesink", "video-sink");
	if ( !vsink ) {
		g_printerr("Failed to create xvimagesink\n");
		return 0;
	}

	/* Set up the pipeline */
	gst_bin_add_many(GST_BIN (pipeline), udpsrc_rtp, udpsrc_rtcp, gstrtpbin, udpsink_rtcp,
			identity, rtpdec, vdec, vconv, vsink, NULL);

	//link udpsrc-rtp to gstrtpbin
    pad = gst_element_get_request_pad(gstrtpbin, "recv_rtp_sink_%d");
	if ( !pad ) {
		g_printerr("Failed to create pad\n");
		return 0;
	}
    res = gst_pad_link(gst_element_get_pad(udpsrc_rtp, "src"), pad);
    if ( GST_PAD_LINK_FAILED(res)  ) {
		g_printerr("Failed to link pads\n");
		return 0;
	}
    gst_object_unref(GST_OBJECT (pad));

	//link udpsrc-rctp to gstrtpbin
#if 0
    pad = gst_element_get_request_pad(gstrtpbin, "recv_rtcp_sink_%d");
 	if ( !pad ) {
		g_printerr("Failed to create pad\n");
		return 0;
	}
    res = gst_pad_link(gst_element_get_pad(udpsrc_rtcp, "src"), pad);
    if ( GST_PAD_LINK_FAILED(res)  ) {
		g_printerr("Failed to link pads\n");
		return 0;
	}
    gst_object_unref(GST_OBJECT (pad));
#endif //0
    pad = gst_element_get_request_pad(gstrtpbin, "recv_rtcp_sink_%d");
 	if ( !pad ) {
		g_printerr("Failed to create pad\n");
		return 0;
	}
    err = gst_element_link_many(udpsrc_rtcp, identity, NULL);
	if ( err==FALSE ) {
		g_printerr("Failed to link elements\n");
		return 0;
	}
    res = gst_pad_link(gst_element_get_pad(identity, "src"), pad);
    if ( GST_PAD_LINK_FAILED(res)  ) {
		g_printerr("Failed to link pads\n");
		return 0;
	}
    gst_object_unref(GST_OBJECT (pad));

	//link udpsink-rctp to gstrtpbin
    pad = gst_element_get_request_pad(gstrtpbin, "send_rtcp_src_%d");
 	if ( !pad ) {
		g_printerr("Failed to create pad\n");
		return 0;
	}
    err = gst_element_link_many(gstrtpbin, udpsink_rtcp, NULL);
	if ( err==FALSE ) {
		g_printerr("Failed to link elements\n");
		return 0;
	}
    gst_object_unref(GST_OBJECT (pad));

    /* listen for newly created pads */
    g_signal_connect (gstrtpbin, "pad-added", G_CALLBACK (cb_new_rtp_recv_src_pad), rtpdec);

	err = gst_element_link_many(rtpdec, vdec, vconv, vsink, NULL);
	if ( err==FALSE ) {
		g_printerr("Failed to link elements\n");
		return 0;
	}

	/* we set the input filename to the udpsrc_rtp element */
	//g_object_set(G_OBJECT (udpsrc_rtp), "location", argv[1], NULL);

	return pipeline;

}

int main(int argc, char **argv)
{
	const gchar *nano_str;
	guint major, minor, micro, nano;
	GMainLoop *loop;
	GstBus *bus;
	GstElement *pipeline;

	//have to put parse_options before gst_init
	//to show mcn_streaming help instead of GStreamer help
	parse_options(argc, argv);

	gst_init(&argc, &argv);

	gst_version(&major, &minor, &micro, &nano);

	if (nano == 1)
		nano_str = "(CVS)";
	else if (nano == 2)
		nano_str = "(Prerelease)";
	else
		nano_str = "";

	g_print("This is " PACKAGE_STRING "\n");
	g_print("This program is linked against GStreamer %d.%d.%d %s\n\n", major,
			minor, micro, nano_str);

	loop = g_main_loop_new(NULL, FALSE);

	//Construct sender pipeline
	pipeline = construct_receiver_pipeline();
	if ( !pipeline ) {
		g_printerr("Failed to create pipeline\n");
		return -1;
	}

	/* we add a message handler */
	bus = gst_pipeline_get_bus(GST_PIPELINE (pipeline));
	gst_bus_add_watch(bus, bus_call, loop);
	gst_object_unref(bus);
	//g_signal_connect (demuxer, "pad-added", G_CALLBACK (on_pad_added), decoder);

	/* Set the pipeline to "playing" state*/
	g_print("Now receiving...\n");
	gst_element_set_state(pipeline, GST_STATE_PLAYING);

	/* Iterate */
	g_print("Running...\n");
	g_main_loop_run(loop);

	/* Out of the main loop, clean up nicely */
	g_print("Returned, stopping receiving\n");
	gst_element_set_state(pipeline, GST_STATE_NULL);

	g_print("Deleting pipeline\n");
	gst_object_unref(GST_OBJECT (pipeline));

	return 0;
}

