/*
 * basictypes.h
 */

#ifndef BASICTYPES_H_
#define BASICTYPES_H_

#include "pin.H"
#include <assert.h>
#include <sstream>
#include <map>

#define VC_SIZE 50
#define EPOCH_INIT_CLOCK 1

#include <malloc.h>
#define PREV_INUSE 0x1
#define IS_MMAPPED 0x2
#define SIZE_BITS PREV_INUSE|IS_MMAPPED
#define chunksize(p) (*(((UINT32 *)p)-1) & ~(SIZE_BITS))

inline int getPSize(void * p){ return chunksize(p) + 4;}

extern int varNum;

typedef UINT32 Epoch;
typedef Epoch* VC;

inline Epoch epoch_new(THREADID tid, UINT32 clock) {
	Epoch e = 0X00000000;
	return (e | tid<<24 | clock);
}

inline int TID(Epoch e) {
	return e>>24; //the higher-8 bit.
}

inline int TClock(Epoch e) {
	return (e & 0X00FFFFFF);
}

inline Epoch epoch_inc(Epoch e) {
	return ++e;
}

inline void printEpoch(Epoch e) {
	printf("%d@%d \n", TClock(e), TID(e));
}

inline VC vc_new() {
	Epoch* vc = new Epoch[VC_SIZE];
	memset(vc, 0, sizeof(Epoch)*VC_SIZE);
	return vc;
}

inline void printVC_Length(VC vc, int length) {
	printf("\n");
	for (int i = 0; i < length; i++) {
		Epoch e = vc[i];
		printf("%d@%d, ", TClock(e), TID(e));
	}
	printf("\n");
}

//vc1 = max(vc1, vc2);
inline void vc_MaxToFirst(VC vc1, VC vc2) {
	for (int i = 0; i < VC_SIZE; i++) {
		if (vc1[i] < vc2[i])
			vc1[i] = vc2[i];
		else
			continue;
	}
}

struct ThreadState{
	THREADID tid;
	Epoch epoch;
	VC vc_C;
};

struct LockState{
	VC vc_L;
};

struct BarrierState{
	VC vc_B;
};

struct VarState{
	Epoch w;
	Epoch r;
	VC vc_R;

	ADDRINT lastINS_W;
	ADDRINT lastINS_R;
	ADDRINT* lastINS_VCR;

	bool race_reported;
	bool isReadShared;
	bool NOACCESS;
};

//VC_INS: stores the last read instruction for each thread.
inline ADDRINT* vc_ins_new() {
	ADDRINT* vc_ins = new ADDRINT[VC_SIZE];
	memset(vc_ins, 0, VC_SIZE);
	return vc_ins;
}

//Shadow memory: two-level mapping structure: PM-SM, proposed in VEE'07: "How to shadow every
//byte of memory used by a program".
//PM has 64K entries of SM. SM has 64K entries. Each SM entries corresponds to one Byte.
//So in total, PM maps to 64K*64KB = 4GB memory (32bit).
//VarState**: a pointer of a array that stores VarState* variables.
struct SM {
	VarState** varArray;
};

static SM PM[65536];
inline bool getSM_IsNewVar(UINT32 addr, VarState** var) {
	SM* sm = &(PM[addr>>16]);
	if (sm->varArray == NULL) {
		sm->varArray = new VarState*[65536];
		memset(sm->varArray, 0, 65536*sizeof(VarState*));
	}

	*var = sm->varArray[addr&0X0000FFFF];
	if (*var == NULL) {
		*var = new VarState();
		(*var)->NOACCESS = false;
		sm->varArray[addr&0X0000FFFF] = *var;
		varNum++;
		return true;
	}

	if ((*var)->NOACCESS) {
		if ((*var)->vc_R != NULL) {
			delete[] (*var)->vc_R;
			(*var)->vc_R = NULL;
		}
		if ((*var)->lastINS_VCR != NULL) {
			delete[] (*var)->lastINS_VCR;
			(*var)->lastINS_VCR = NULL;
		}
		return true;
	}
	return false;
}

//Delete the whole shadow memory when analysis completes.
inline void deletePMSM() {
	SM* sm;
	VarState* var;
	for (int i = 0; i < 65536; i++) {
		sm = &(PM[i]);
		if (sm->varArray == NULL)
			continue;
		for (int j = 0; j < 65536; j++) {
			var = (sm->varArray)[j];
			if (var != NULL)
				delete var;
		}
	}
}

//Delete the shadow memory that is not used (e.g. free()) in the program.
inline void deleteSM_NotUsed(UINT32 addr_head, UINT32 addr_tail) {
	SM* sm = &(PM[addr_head>>16]);
	if (sm->varArray == NULL)
		return;

	VarState* var = NULL;
	for (UINT32 addr = addr_head; addr <= addr_tail; addr++) {
		var = (sm->varArray)[addr&0X0000FFFF];
		if (var != NULL)
			var->NOACCESS = true;
	}
}

inline void deleteSMKey(void* key)
{
	int size = getPSize(key)-1; 

	int n=(UINT32)(((UINT32)key + size)>>16) -(UINT32)((UINT32)key>>16);
	if(n==0)
	{
		deleteSM_NotUsed((UINT32)key, (UINT32)key+size);
	}
	else
	{
		//(1)
		deleteSM_NotUsed((UINT32) key, (UINT32)key| 0x0000FFFF);
		//(2)
		for(int i=1;i<n;i++)
		{
			deleteSM_NotUsed(
					(((UINT32) key)+(i<<16)) & 0xFFFF0000,
					(((UINT32) key)+(i<<16)) | 0x0000FFFF
					);
		}
		//(3)
		deleteSM_NotUsed(((UINT32)key + size) & 0xFFFF0000, (UINT32)key + size);
	}
}

inline void get_statement_location(ADDRINT ins, INT32 & lineNum, std::string & fileName)
{
	PIN_LockClient();
	PIN_GetSourceLocation(ins, NULL, &lineNum, &fileName);
	PIN_UnlockClient();
}

FILE *raceFile;
PIN_LOCK raceFile_Lock;
static int raceNum = 0;
inline void report_race(char* race_type, THREADID tid1, ADDRINT ins1, Epoch e1,
		THREADID tid2, ADDRINT ins2, Epoch e2, ADDRINT mem) {
	RTN rtn1, rtn2;
	PIN_LockClient();
	rtn1 = RTN_FindByAddress(ins1);
	rtn2 = RTN_FindByAddress(ins2);
	std::string rtn1Name, rtn2Name;
	if (RTN_Valid(rtn1) && RTN_Valid(rtn2)) {
		rtn1Name = RTN_Name(rtn1);
		rtn2Name = RTN_Name(rtn2);
	} else {
		rtn1Name = "Null";
		rtn2Name = "Null";
	}
	PIN_UnlockClient();

	INT32 lineNum1, lineNum2;
	std::string fileName1, fileName2;
	get_statement_location(ins1, lineNum1, fileName1);
	get_statement_location(ins2, lineNum2, fileName2);

	int strHead1 = fileName1.size()/2;
	int strHead2 = fileName2.size()/2;
	fileName1 = fileName1.substr(strHead1, fileName1.size()-strHead1);
	fileName2 = fileName2.substr(strHead2, fileName2.size()-strHead2);

	GetLock(&raceFile_Lock, 1);
	raceNum++;
	fprintf(raceFile, "Race %d:\n", raceNum);
	fprintf(raceFile, "Type %s:\n", race_type);
	fprintf(raceFile, "Memory %X\n", mem);
	fprintf(raceFile, "Thread %d-%d\n", tid1, tid2);
	fprintf(raceFile, "Epoch %d@%d-%d@%d\n", TClock(e1), TID(e1), TClock(e2), TID(e2));
	fprintf(raceFile, "RTN %s-%s\n", (char*)rtn1Name.c_str(), (char*)rtn2Name.c_str());
	fprintf(raceFile, "Statement %s_%d-%s_%d\n", (char*)fileName1.c_str(), lineNum1, (char*)fileName2.c_str(), lineNum2);
	ReleaseLock(&raceFile_Lock);
}

/*!
 *  Print out help message.
 */
INT32 Usage()
{
    cerr << "This tool prints out the number of dynamically executed " << endl <<
            "instructions, basic blocks and threads in the application." << endl << endl;

    cerr << KNOB_BASE::StringKnobSummary() << endl;

    return -1;
}

inline bool IsTraceIgnoreImage(IMG img) {
	string name = IMG_Name(img);
	if (name.find("ld-") != string::npos)
		return true;
	if (name.find("/lib/i386-linux-gnu") != string::npos)
		return true;
	if (name.find("/lib/x86_64-linux-gnu") != string::npos)
		return true;
	return false;
}

inline bool IsTraceIgnoreRoutine(RTN rtn) {
    string name = RTN_Name(rtn);
    if (name.find_first_of("_") == 0)
    	return true;
    return false;
}

extern map<pthread_t, ThreadState*> pthread_threadstate_map;
extern map<pthread_mutex_t*, LockState*> pthread_lockstate_map;
extern map<pthread_barrier_t*, BarrierState*> pthread_barrierstate_map;

void deleteStateMap() {
	pthread_threadstate_map.clear();
	pthread_lockstate_map.clear();
	pthread_barrierstate_map.clear();
}

#endif /* BASICTYPES_H_ */
